<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
// phpcs:disable Generic.Commenting.DocComment.MissingShort
/**
 * Handles adding fields to the REST fields.
 *
 * @package LD_Organization\Hooks
 */

namespace LD_Organization\Hooks;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Adds fields to the essay rest API object that is needed in the frontend.
 *
 * @package LD_Organization\Hooks
 */
class EssayRestFields extends AbstractHook {
	/**
	 * Handles adding the fields to the rest API object for sfwd-essay.
	 *
	 * @return void
	 * @uses learndash_get_post_type_slug
	 */
	final public function add_rest_fields(): void {
		$essay_post_type = learndash_get_post_type_slug( 'essay', 'string' );
		register_rest_field(
			$essay_post_type,
			'essay_details',
			array(
				'get_callback' => array(
					$this,
					'get_extra_essay_data',
				),
			)
		);
		register_rest_field(
			$essay_post_type,
			'user_details',
			array(
				'get_callback' => array(
					$this,
					'get_essay_user_information',
				),
			)
		);
	}

	/**
	 * Handles fetching additional fields for essays Rest API object.
	 *
	 * @param array $essay The prepared essay from the Rest API object.
	 *
	 * @return array
	 */
	final public function get_extra_essay_data( array $essay ): array {
		if ( ! empty( $essay['id'] ) ) {
			$essay_details              = learndash_get_essay_details( $essay['id'] );
			$essay_details['is_upload'] = false;
			if ( ! empty( get_post_meta( $essay['id'], 'upload', true ) ) ) {
				$essay_details['is_upload'] = true;
				$essay_details['upload']    = get_post_meta( $essay['id'], 'upload', true );
			}
			if ( ! empty( get_post_meta( $essay['id'], 'question_id', true ) ) ) {
				$question_id = get_post_meta( $essay['id'], 'question_id', true );
				$fields      = learndash_get_question_pro_fields( $question_id, array( 'title', 'question' ) );
				if ( ! empty( $fields ) ) {
					$essay_details['question'] = $fields;
				}
			}
			if ( ! empty( $essay['author'] ) ) {
				$essay_quiz_details = learndash_get_user_quiz_entry_for_essay( $essay['id'], $essay['author'] );
				if ( ! empty( $essay_quiz_details ) ) {
					$essay_details['quiz'] = $essay_quiz_details;
				}
			}
			$this->add_course_details( $essay_details, $essay );

			return $essay_details;
		}

		return array();
	}

	/**
	 * Handles adding course details to the essay.
	 *
	 * @param array $essay_details The essay details array.
	 * @param array $essay The essay object from the Rest API.
	 *
	 * @return void
	 */
	private function add_course_details( array &$essay_details, array $essay ): void {
		if ( ! empty( $essay['course'] ) && is_int( $essay['course'] ) ) {
			$course = get_post( $essay['course'] );
			if ( ! empty( $course ) ) {
				$essay_details['course'] = array(
					'ID'    => $course->ID,
					'title' => $course->post_title,
				);
			}
		}
		if ( ! empty( $essay['lesson'] ) && is_int( $essay['lesson'] ) ) {
			$course = get_post( $essay['lesson'] );
			if ( ! empty( $course ) ) {
				$essay_details['lesson'] = array(
					'ID'    => $course->ID,
					'title' => $course->post_title,
				);
			}
		}
		if ( ! empty( $essay['topic'] ) && is_int( $essay['topic'] ) ) {
			$course = get_post( $essay['topic'] );
			if ( ! empty( $course ) ) {
				$essay_details['topic'] = array(
					'ID'    => $course->ID,
					'title' => $course->post_title,
				);
			}
		}
	}

	/**
	 * Handles checking if the author is defined on the essay, and adds the user data if valid.
	 *
	 * @param array $essay The prepared essay from the Rest API object.
	 *
	 * @return array
	 */
	final public function get_essay_user_information( array $essay ): array {
		if ( ! empty( $essay['author'] ) ) {
			$user = get_user_by( 'ID', $essay['author'] );
			if ( ! empty( $user ) ) {
				return array(
					'user_id'    => $user->ID,
					'first_name' => $user->first_name,
					'last_name'  => $user->last_name,
				);
			}
		}

		return array();
	}

	/**
	 * @inheritDoc
	 */
	final public function get_actions(): array {
		return array(
			array(
				'hook'     => 'rest_api_init',
				'callable' => array( $this, 'add_rest_fields' ),
				'priority' => 99,
				'num_args' => 0,
			),
		);
	}

	/**
	 * @inheritDoc
	 */
	final public function get_filters(): array {
		return array();
	}
}
