<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase

/**
 * AbstractCronJob class is an abstract class that provides some sane defaults for registering cron jobs.
 *
 * @package LD_Organization\Cron
 */

namespace LD_Organization\Cron;

/**
 * Abstract class for cron jobs.
 *
 * @package LD_Organization\Cron
 */
abstract class AbstractCronJob {
	/**
	 * The hook name for the cron job.
	 *
	 * @var string
	 */
	protected string $cron_hook;

	/**
	 * The interval for the cron job.
	 *
	 * @var string
	 */
	protected string $cron_interval = 'daily';

	/**
	 * The arguments for the cron job.
	 *
	 * @var array
	 */
	protected array $cron_args = array();

	/**
	 * Initializes the cron jobs
	 *
	 * @param string $cron_interval The interval for the cron job.
	 * @param array $cron_args The arguments for the cron job.
	 */
	public function __construct( string $cron_interval, array $cron_args ) {
		$this->cron_hook = static::class . '_cron_job';
		$this->cron_interval = $cron_interval;
		$this->cron_args     = $cron_args;
		add_action( 'init', array( $this, 'schedule_cron_job' ) );
		add_action( $this->cron_hook, array( $this, 'do_cron_job' ) );
	}


	/**
	 * Schedules the cron job.
	 *
	 * @return void
	 */
	final public function schedule_cron_job(): void {
		if ( ! wp_next_scheduled( $this->cron_hook, $this->cron_args ) ) {
			wp_schedule_event( time(), $this->cron_interval, $this->cron_hook, $this->cron_args );
		}
	}


	/**
	 * This will be called when the cron job is run.
	 *
	 * @return void
	 */
	abstract public function do_cron_job(): void;
}
