<?php
/**
 * All learndash course helper functions and other functions that can be useful are declared here.
 *
 * @package LD_Organization
 * @since 0.3.0
 */

namespace LD_Organization;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use InvalidArgumentException;
use WP_User_Query;

/**
 * Handles calculating the course progress for the specified course.
 *
 * @param int $group_id The groups users to check against.
 * @param int $course_id The course ID to calculate progress for.
 *
 * @return array{in_progress:int,progress:int,passed:int,not_passed:int} The progress array.
 * @throws InvalidArgumentException When either the group or course ID is invalid.
 * @since 0.3.0
 */
function calculate_groups_course_progress( int $group_id, int $course_id ): array {
	$passed     = 0;
	$not_passed = 0;
	$started    = 0;

	// First we check if all arguments are valid.
	if ( ! is_numeric( $course_id ) || ! learndash_is_course_post( $course_id ) ) {
		throw new InvalidArgumentException( 'Please pass an course ID that belongs to a course.' );
	}

	if ( ! is_numeric( $group_id ) || ! learndash_is_group_post( $group_id ) ) {
		throw new InvalidArgumentException( 'Please pass an group ID that belongs to a group.' );
	}

	// Then we get the groups associated with the course.
	$groups = learndash_get_course_groups( $course_id );

	// Check if the group is in the course.
	if ( ! in_array( $group_id, $groups, true ) ) {
		throw new InvalidArgumentException( 'The group is not in the requested course' );
	}

	// Then we perform some merging to get the users in the course and in the group.
	$course_users = learndash_get_course_groups_users_access( $course_id );

	$group_users = learndash_get_groups_user_ids( $group_id );

	$users_to_check = array();
	if ( ! empty( $course_users ) && ! empty( $group_users ) ) {
		foreach ( $course_users as $course_user ) {
			if ( in_array( $course_user, $group_users, true ) ) {
				$users_to_check[] = $course_user;
			}
		}
	}

	$users_to_check = array_unique( $users_to_check );

	foreach ( $users_to_check as $user_id ) {
		$temp_progress = learndash_user_get_course_progress( $user_id, $course_id, 'summary' );
		if ( empty( $temp_progress ) || ! isset( $temp_progress['completed'], $temp_progress['total'], $temp_progress['status'] ) ) {
			continue;
		}

		$done = $temp_progress['completed'] === $temp_progress['total'] || 'done' === $temp_progress['status'];
		if ( $done ) {
			$passed++;
			$started++;
		} else {
			$not_passed++;
			if ( 'in_progress' === $temp_progress['status'] ) {
				$started++;
			}
		}
	}

	$user_count = count( $users_to_check );
	$progress   = 0 === $user_count ? 0 : $passed / $user_count * 100;

	return array(
		'in_progress' => handle_erroneous_values( $started ),
		'not_passed'  => handle_erroneous_values( $not_passed ),
		'passed'      => handle_erroneous_values( $passed ),
		'progress'    => handle_erroneous_values( $progress ),
	);
}


/**
 * Fetches all courses ids.
 *
 * @return int[]
 */
function get_all_course_ids(): array {
	$course_slug = learndash_get_post_type_slug( 'course' );

	return get_posts(
		array(
			'post_type'   => $course_slug,
			'posts',
			'numberposts' => - 1,
			'fields'      => 'ids',
		)
	);
}

/**
 * This function can be used when a course has too many users for WP_User_Query includes to handle.
 *
 * @param int $course_id The course ID.
 *
 * @return array
 */
function get_users_from_course_no_query( int $course_id ): array {
	$course_user_ids = array();

	if ( empty( $course_id ) ) {
		return $course_user_ids;
	}

	$defaults = array(
		// By default WP_User_Query will return ALL users. Strange.
		'fields' => 'ID',
	);

	$course_price_type = learndash_get_course_meta_setting( $course_id, 'course_price_type' );

	if ( 'open' === $course_price_type ) {
		return ( new WP_User_Query( $defaults ) )->get_results();
	}

	if ( true === learndash_use_legacy_course_access_list() ) {
		$course_access_list = learndash_get_course_meta_setting( $course_id, 'course_access_list' );
		$course_user_ids    = array_merge( $course_user_ids, $course_access_list );
	}

	$course_access_users = learndash_get_course_users_access_from_meta( $course_id );
	$course_user_ids     = array_merge( $course_user_ids, $course_access_users );

	$course_groups_users = learndash_get_course_groups_users_access( $course_id );
	$course_user_ids     = array_merge( $course_user_ids, $course_groups_users );

	if ( ! empty( $course_user_ids ) ) {
		$course_user_ids = array_unique( $course_user_ids );
	}

	$course_expired_access_users = learndash_get_course_expired_access_from_meta( $course_id );
	if ( ! empty( $course_expired_access_users ) ) {
		$course_user_ids = array_diff( $course_user_ids, $course_expired_access_users );
	}

	return array_map( 'absint', $course_user_ids );
}
