<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName

namespace LD_Organization\Admin;

use Exception;
use LD_Organization\Organization;

use const LD_Organization\LD_ORG_BASE_URL;

/**
 * Handles registering the Organization Completions page.
 *
 * @package LD_Organization\Admin
 * @since 0.21.0
 */
class OrganizationCompletions extends AbstractAdminPage {

	/**
	 * @inheritDoc
	 */
	final public function register_pages(): void {
		add_submenu_page(
			'edit.php?post_type=organization',
			__( 'Organization Completions', 'ld-organization' ),
			__( 'Organization Completions', 'ld-organization' ),
			'manage_options',
			'ld-organization-completions',
			array( $this, 'page_callback' )
		);
	}

	/**
	 * @inheritDoc
	 */
	final public function handle_post(): void {
		// Nothing to do here.
	}

	/**
	 * @inheritDoc
	 */
	final public function enqueue_scripts( string $hook ): void {
		if ( 'organization_page_ld-organization-completions' !== $hook ) {
			return;
		}
		wp_enqueue_script( 'admin-vendor', trailingslashit( LD_ORG_BASE_URL ) . 'js/dist/admin-vendor.min.js', array(), '1', true );
		wp_enqueue_script( 'datatables-bundle', 'https://cdn.datatables.net/v/dt/jszip-3.10.1/dt-2.1.8/b-3.2.0/b-html5-3.2.0/fc-5.0.4/fh-4.0.1/sb-1.8.1/datatables.min.js', array(), '2.1.8', false );
		wp_enqueue_style( 'datatables-bundle', 'https://cdn.datatables.net/v/dt/jszip-3.10.1/dt-2.1.8/b-3.2.0/b-html5-3.2.0/fc-5.0.4/fh-4.0.1/sb-1.8.1/datatables.min.css', array(), '2.1.8' );
	}

	final public function old_page_callback(): void {
		$all_organizations = get_posts(
			array(
				'post_type'   => 'organization',
				'post_status' => 'publish',
				'fields'      => 'ids',
				'numberposts' => - 1,
			)
		);

		$organization_data = array();
		$years             = array();
		foreach ( $all_organizations as $organization_id ) {
			try {
				$organization = Organization::with_id( $organization_id );
				if ( ! $organization->has_organization() ) {
					continue;
				}
				$completions_raw = $organization->get_all_completions( true );
				if ( empty( $completions_raw ) ) {
					continue;
				}
				$final_completions = array();
				$courses_years     = array();
				foreach ( $completions_raw as $year => $completion ) {
					$total_completed = 0;
					foreach ( $completion['courses'] as $course_id => $course_completions ) {
						$total_completed                    += $course_completions;
						$course_name                         = get_the_title( $course_id );
						$completion['courses'][ $course_id ] = array(
							'id'        => $course_id,
							'title'     => $course_name,
							'completed' => $course_completions,
						);
						if ( ! isset( $courses_years[ $course_id ] ) ) {
							$courses_years[ $course_id ]          = array(
								'id'    => $course_id,
								'title' => $course_name,
							);
							$courses_years[ $course_id ]['years'] = array();
						}
						$courses_years[ $course_id ]['years'][ $year ] = $course_completions;
					}
					$final_completions[ $year ] = array(
						'total_completed' => $total_completed,
						'courses'         => $completion['courses'],
					);
				}
				$organization_data[ $organization_id ] = array(
					'id'          => $organization->get_organization()->ID,
					'title'       => $organization->get_organization()->post_title,
					'completions' => $final_completions,
					'courses'     => $courses_years,
				);
				foreach ( $final_completions as $year => $completions ) {
					if ( ! in_array( $year, $years, true ) ) {
						$years[] = $year;
					}
				}
			} catch ( Exception $e ) {
				logger( 'OrganizationCompletions' )->error( 'Failed to fetch organization data.', array( 'error' => $e->getMessage() ) );
				continue;
			}
		}
		$years = array_unique( $years );
		usort(
			$years,
			static function ( $a, $b ) {
				return absint( $b ) - absint( $a );
			}
		);
		$years = array_slice( $years, 0, 4 );

		?>
		<script>
			document.addEventListener('alpine:init', () => {
				Alpine.store('main', {
					tables: {},
					addTable(id, table) {
						this.tables[id] = table;
					}
				});
			});
		</script>
		<div class="wrap" x-data="{
			selectedOrganization: '0'
		}" x-init="$watch('selectedOrganization', value => {
			if (value === '0') {
				return;
			}
			$nextTick(() => {
				const table = Alpine.store('main').tables[value];
				if (!table) {
					return;
				}
				table.draw();
			});
		})">
			<h2><?php esc_html_e( 'Organization Completions', 'ld-organization' ); ?></h2>
			<?php settings_errors(); ?>
			<a href="<?php echo esc_url( add_query_arg( array( 'archive' => 'false' ), admin_url( 'edit.php?post_type=organization&page=ld-organization-completions' ) ) ); ?>" class="button button-secondary"><?php esc_html_e( 'View Monthly Completions', 'ld-organization' ); ?></a>
			<style>
				.wppb-serial-notification {
					display: none;
				}

				.select-wrapper {
					display: flex;
					flex-direction: column;
					justify-content: center;
					padding: 1rem 0;
				}

				[x-cloak] {
					display: none;
				}
			</style>
			<script>
				const defaultTableOptions = {
					language: {
						url: '//cdn.datatables.net/plug-ins/2.1.8/i18n/fi.json',
					},
					layout: {
						topStart: {
							buttons: [
								{
									extend: 'pageLength',
								},
								{
									extend: 'excel',
									text: '<?php esc_html_e( 'Export to Excel', 'ld-organization' ); ?>',
								},
							],
						}
					}
				};
			</script>
			<div class="select-wrapper">
				<label
					for="ld-organization-completions-organization"><?php esc_html_e( 'Select an organization to see course specific completions', 'ld-organization' ); ?></label>
				<div class="select-reset-wrapper">
					<select id="ld-organization-completions-organization" x-model="selectedOrganization">
						<option
							value="0"><?php esc_html_e( 'Select an organization', 'ld-organization' ); ?></option>
						<?php foreach ( $organization_data as $organization_id => $organization ) : ?>
							<option
								value="<?php echo esc_attr( $organization_id ); ?>"><?php echo esc_html( $organization['title'] ); ?></option>
						<?php endforeach; ?>
					</select>
					<button type="button" class="button button-secondary" @click="selectedOrganization = '0'" x-cloak
							x-show="selectedOrganization !== '0'"><?php esc_html_e( 'Reset', 'ld-organization' ); ?></button>
				</div>
			</div>
			<div class="main-table-wrapper" x-show="selectedOrganization === '0'">
				<table id="ld-organization-completions">
					<thead>
					<td><?php esc_html_e( 'Organization', 'ld-organization' ); ?></td>
					<?php foreach ( $years as $year ) : ?>
						<td><?php echo esc_html( $year ); ?></td>
					<?php endforeach; ?>
					</thead>
					<tbody>
					<?php foreach ( $organization_data as $organization_id => $organization ) : ?>
						<tr>
							<td><?php echo esc_html( $organization['title'] ); ?></td>
							<?php foreach ( $years as $year ) : ?>
								<td><?php echo esc_html( $organization['completions'][ $year ]['total_completed'] ?? 0 ); ?></td>
							<?php endforeach; ?>
						</tr>
					<?php endforeach; ?>
					</tbody>
				</table>
				<script>
					new DataTable('#ld-organization-completions', defaultTableOptions);
				</script>
			</div>
			<?php foreach ( $organization_data as $organization_id => $organization ) : ?>
				<div class="selected-wrapper"
					x-show="selectedOrganization === '<?php echo esc_attr( $organization_id ); ?>'"
					x-cloak
				>
					<h3>
						<?php esc_html_e( 'Organization: ', 'ld-organization' ); ?>
						<?php echo esc_html( $organization['title'] ); ?>
					</h3>
					<table id="ld-organization-completions-<?php echo esc_attr( $organization_id ); ?>">
						<thead>
						<td><?php esc_html_e( 'Course', 'ld-organization' ); ?></td>
						<?php foreach ( $years as $year ) : ?>
							<td><?php echo esc_html( $year ); ?></td>
						<?php endforeach; ?>
						</thead>
						<tbody>
						<?php foreach ( $organization['courses'] as $course_id => $course ) : ?>
							<?php
							if ( empty( $course['title'] ) ) {
								continue;
							}
							?>
							<tr>
								<td><?php echo esc_html( $course['title'] ); ?></td>
								<?php foreach ( $years as $year ) : ?>
									<td><?php echo esc_html( $course['years'][ $year ] ?? 0 ); ?></td>
								<?php endforeach; ?>
							</tr>
						<?php endforeach; ?>
						</tbody>
						<tfoot>
						<tr>
							<td style="font-weight: 800"><?php esc_html_e( 'Total', 'ld-organization' ); ?></td>
							<?php foreach ( $years as $year ) : ?>
								<td style="font-weight: 600"><?php echo esc_html( $organization['completions'][ $year ]['total_completed'] ?? 0 ); ?></td>
							<?php endforeach; ?>
						</tr>
						</tfoot>
					</table>
					<script>
						document.addEventListener('alpine:init', () => {
							const table = new DataTable('#ld-organization-completions-<?php echo esc_attr( $organization_id ); ?>', defaultTableOptions);
							Alpine.store('main').addTable('<?php echo esc_attr( $organization_id ); ?>', table);
						});
					</script>
				</div>
			<?php endforeach; ?>
		</div>
		<?php
	}

	/**
	 * Handles outputting the html.
	 *
	 * @return void All output is echoed.
	 */
	final public function page_callback(): void {
		$is_archive = isset( $_GET['archive'] ) && $_GET['archive'] === 'true';
		if ( $is_archive ) {
			$this->old_page_callback();
			return;
		}
		$all_organizations = get_posts(
			array(
				'post_type'   => 'organization',
				'post_status' => 'publish',
				'fields'      => 'ids',
				'numberposts' => - 1,
			)
		);

		$organization_data = array();
		$years             = array();
		foreach ( $all_organizations as $organization_id ) {
			try {
				$organization = Organization::with_id( $organization_id );
				if ( ! $organization->has_organization() ) {
					continue;
				}
				$completions_raw = $organization->get_all_completions();
				if ( empty( $completions_raw ) ) {
					continue;
				}
				$final_completions = array();
				$courses_years     = array();
				foreach ( $completions_raw as $year => $completion ) {
					$total_completed = 0;
					foreach ( $completion['courses'] as $course_id => $course_completions ) {
						$total_completed                    += $course_completions;
						$course_name                         = get_the_title( $course_id );
						$completion['courses'][ $course_id ] = array(
							'id'        => $course_id,
							'title'     => $course_name,
							'completed' => $course_completions,
						);
						if ( ! isset( $courses_years[ $course_id ] ) ) {
							$courses_years[ $course_id ]          = array(
								'id'    => $course_id,
								'title' => $course_name,
							);
							$courses_years[ $course_id ]['years'] = array();
						}
						$courses_years[ $course_id ]['years'][ $year ] = $course_completions;
					}
					// Normalize months to always have 01-12 keys with integer values.
					$months_raw   = isset( $completion['months'] ) && is_array( $completion['months'] ) ? $completion['months'] : array();
					$months_final = array();
					for ( $m = 1; $m <= 12; $m++ ) {
						$month_key                  = str_pad( (string) $m, 2, '0', STR_PAD_LEFT );
						$months_final[ $month_key ] = isset( $months_raw[ $month_key ] ) ? absint( $months_raw[ $month_key ] ) : 0;
					}
					$final_completions[ $year ] = array(
						'total_completed' => $total_completed,
						'courses'         => $completion['courses'],
						'months'          => $months_final,
					);
				}
				$organization_data[ $organization_id ] = array(
					'id'          => $organization->get_organization()->ID,
					'title'       => $organization->get_organization()->post_title,
					'completions' => $final_completions,
					'courses'     => $courses_years,
				);
				foreach ( $final_completions as $year => $completions ) {
					if ( ! in_array( $year, $years, true ) ) {
						$years[] = $year;
					}
				}
			} catch ( Exception $e ) {
				logger( 'OrganizationCompletions' )->error( 'Failed to fetch organization data.', array( 'error' => $e->getMessage() ) );
				continue;
			}
		}
		$years = array_unique( $years );
		usort(
			$years,
			static function ( $a, $b ) {
				return absint( $b ) - absint( $a );
			}
		);
		$years = array_slice( $years, 0, 4 );

		?>
		<script>
			document.addEventListener('alpine:init', () => {
				Alpine.store('main', {
					tables: {},
					addTable(id, table) {
						this.tables[id] = table;
					}
				});
			});
		</script>
		<div class="wrap" x-data="{
			selectedOrganization: '0'
		}" x-init="$watch('selectedOrganization', value => {
			if (value === '0') {
				return;
			}
			$nextTick(() => {
				const table = Alpine.store('main').tables[value];
				if (!table) {
					return;
				}
				table.draw();
			});
		})">
			<h2><?php esc_html_e( 'Organization Completions', 'ld-organization' ); ?></h2>
			<?php settings_errors(); ?>
			<a href="<?php echo esc_url( add_query_arg( array( 'archive' => 'true' ), admin_url( 'edit.php?post_type=organization&page=ld-organization-completions' ) ) ); ?>" class="button button-secondary"><?php esc_html_e( 'View Archived Completions', 'ld-organization' ); ?></a>
			<style>
				.wppb-serial-notification {
					display: none;
				}

				.select-wrapper {
					display: flex;
					flex-direction: column;
					justify-content: center;
					padding: 1rem 0;
				}

				[x-cloak] {
					display: none;
				}
			</style>
			<script>
				const defaultTableOptions = {
					language: {
						url: '//cdn.datatables.net/plug-ins/2.1.8/i18n/fi.json',
					},
					layout: {
						topStart: {
							buttons: [
								{
									extend: 'pageLength',
								},
								{
									extend: 'excel',
									text: '<?php esc_html_e( 'Export to Excel', 'ld-organization' ); ?>',
								},
							],
						}
					}
				};
			</script>
			<div class="select-wrapper">
				<label
					for="ld-organization-completions-organization"><?php esc_html_e( 'Select an organization to see monthly completions', 'ld-organization' ); ?></label>
				<div class="select-reset-wrapper">
					<select id="ld-organization-completions-organization" x-model="selectedOrganization">
						<option
							value="0"><?php esc_html_e( 'Select an organization', 'ld-organization' ); ?></option>
						<?php foreach ( $organization_data as $organization_id => $organization ) : ?>
							<option
								value="<?php echo esc_attr( $organization_id ); ?>"><?php echo esc_html( $organization['title'] ); ?></option>
						<?php endforeach; ?>
					</select>
					<button type="button" class="button button-secondary" @click="selectedOrganization = '0'" x-cloak
							x-show="selectedOrganization !== '0'"><?php esc_html_e( 'Reset', 'ld-organization' ); ?></button>
				</div>
			</div>
			<div class="main-table-wrapper" x-show="selectedOrganization === '0'">
				<table id="ld-organization-completions">
					<thead>
					<td><?php esc_html_e( 'Organization', 'ld-organization' ); ?></td>
					<?php foreach ( $years as $year ) : ?>
						<td><?php echo esc_html( $year ); ?></td>
					<?php endforeach; ?>
					</thead>
					<tbody>
					<?php foreach ( $organization_data as $organization_id => $organization ) : ?>
						<tr>
							<td><?php echo esc_html( $organization['title'] ); ?></td>
							<?php foreach ( $years as $year ) : ?>
								<td><?php echo esc_html( $organization['completions'][ $year ]['total_completed'] ?? 0 ); ?></td>
							<?php endforeach; ?>
						</tr>
					<?php endforeach; ?>
					</tbody>
				</table>
				<script>
					new DataTable('#ld-organization-completions', defaultTableOptions);
				</script>
			</div>
			<?php foreach ( $organization_data as $organization_id => $organization ) : ?>
				<div class="selected-wrapper"
					x-show="selectedOrganization === '<?php echo esc_attr( $organization_id ); ?>'"
					x-cloak
				>
					<h3>
						<?php esc_html_e( 'Organization: ', 'ld-organization' ); ?>
						<?php echo esc_html( $organization['title'] ); ?>
					</h3>
					<table id="ld-organization-completions-<?php echo esc_attr( $organization_id ); ?>">
						<thead>
						<td><?php esc_html_e( 'Year', 'ld-organization' ); ?></td>
						<?php for ( $i = 1; $i <= 12; $i++ ) : ?>
							<?php $month_label = date_i18n( 'M', mktime( 0, 0, 0, $i, 1 ) ); ?>
							<td><?php echo esc_html( $month_label ); ?></td>
						<?php endfor; ?>
						<td><?php esc_html_e( 'Total', 'ld-organization' ); ?></td>
						</thead>
						<tbody>
						<?php foreach ( $years as $year ) : ?>
							<tr>
								<td><?php echo esc_html( $year ); ?></td>
								<?php
								$total_for_year = 0;
								for ( $i = 1; $i <= 12; $i++ ) :
									$month_key       = str_pad( (string) $i, 2, '0', STR_PAD_LEFT );
									$val             = isset( $organization['completions'][ $year ]['months'][ $month_key ] ) ? (int) $organization['completions'][ $year ]['months'][ $month_key ] : 0;
									$total_for_year += $val;
									?>
									<td><?php echo esc_html( $val ); ?></td>
								<?php endfor; ?>
								<td style="font-weight: 600"><?php echo esc_html( $total_for_year ); ?></td>
							</tr>
						<?php endforeach; ?>
						</tbody>
					</table>
					<script>
						document.addEventListener('alpine:init', () => {
							const table = new DataTable('#ld-organization-completions-<?php echo esc_attr( $organization_id ); ?>', defaultTableOptions);
							Alpine.store('main').addTable('<?php echo esc_attr( $organization_id ); ?>', table);
						});
					</script>
				</div>
			<?php endforeach; ?>
		</div>
		<?php
	}
}
