<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName

namespace LD_Organization\Admin;

use Exception;
use LD_Organization\Organization;

use const LD_Organization\LD_ORG_BASE_URL;

/**
 * Handles registering the Organization Activity page.
 *
 * @package LD_Organization\Admin
 * @since 0.22.0
 */
class ActivityAdmin extends AbstractAdminPage {

	/**
	 * @inheritDoc
	 */
	final public function register_pages(): void {
		add_submenu_page(
			'edit.php?post_type=organization',
			__( 'User activity', 'ld-organization' ),
			__( 'User activity', 'ld-organization' ),
			'manage_options',
			'ld-organization-activity',
			array( $this, 'page_callback' )
		);
	}

	/**
	 * @inheritDoc
	 */
	final public function handle_post(): void {
		// Nothing to do here.
	}

	/**
	 * @inheritDoc
	 */
	final public function enqueue_scripts( string $hook ): void {
		if ( 'organization_page_ld-organization-activity' !== $hook ) {
			return;
		}

		// Alpine & Other Vendor Scripts
		wp_enqueue_script( 'admin-vendor', trailingslashit( LD_ORG_BASE_URL ) . 'js/dist/admin-vendor.min.js', array(), '1', true );

		// Luxon
		wp_enqueue_script( 'vendor-luxon', 'https://cdn.jsdelivr.net/npm/luxon@3.6.1/build/global/luxon.min.js', array(), '3', true );

		// Datatables
		wp_enqueue_script( 'datatables-bundle', 'https://cdn.datatables.net/v/dt/dt-2.2.2/b-3.2.2/date-1.5.5/sp-2.3.3/sl-3.0.0/datatables.min.js', array( 'vendor-luxon' ), '2.1.8', false );
		wp_enqueue_style( 'datatables-bundle', 'https://cdn.datatables.net/v/dt/dt-2.2.2/b-3.2.2/date-1.5.5/sp-2.3.3/sl-3.0.0/datatables.min.css', array(), '2.1.8' );

		// Notyf
		wp_enqueue_script( 'notyf', 'https://cdn.jsdelivr.net/npm/notyf@3/notyf.min.js', array(), '3', true );
		wp_enqueue_style( 'notyf', 'https://cdn.jsdelivr.net/npm/notyf@3/notyf.min.css', array(), '3' );
	}

	/**
	 * Handles outputting the html.
	 *
	 * @return void All output is echoed.
	 */
	final public function page_callback(): void {
		$all_organizations = get_posts(
			array(
				'post_type'   => 'organization',
				'post_status' => 'publish',
				'numberposts' => - 1,
			)
		);

		?>
		<div class="wrap" x-data="{
			selectedOrganization: '0',
			courses: [],
			selectedCourse: '0',
			loading: false,
			init() {
				$watch('selectedOrganization', async (value) => {
					sessionStorage.setItem('selectedOrganization', value);
					if (value === '0') {
						this.courses = [];
						this.selectedCourse = '0';
						return;
					}
					this.selectedCourse = '0';
					this.loading = true;
					const response = await fetch('<?php echo esc_url( admin_url( 'admin-ajax.php' ) ); ?>', {
						method: 'POST',
						body: new URLSearchParams({
							action: 'fetch_org_courses',
							organization_id: value,
							nonce: '<?php echo esc_js( wp_create_nonce( 'ld-organization-activity-nonce' ) ); ?>',
						}),
					});
					const responseData = await response.json();
					if (responseData.success) {
						this.courses = responseData.data.courses;
					} else {
						this.courses = [];
						new Notyf().error(responseData.data.message ?? '<?php esc_html_e( 'An error occurred', 'ld-organization' ); ?>');
					}
					this.loading = false;
				});
				$watch('selectedCourse', async (value) => {
					if (value === '0' || this.selectedOrganization === '0') {
						return;
					}
					Alpine.store('table').renderTable(this.selectedOrganization, value);
				});
				if (sessionStorage.getItem('selectedOrganization')) {
					this.selectedOrganization = sessionStorage.getItem('selectedOrganization');
				}
			}
		}">
			<h2><?php esc_html_e( 'User activity', 'ld-organization' ); ?></h2>
			<?php settings_errors(); ?>	
			<div class="select-wrapper">
				<label
					for="ld-organization-activity-organization"><?php esc_html_e( 'Select an organization to see user activity in a course', 'ld-organization' ); ?></label>
				<div class="select-data-wrapper">
					<select id="ld-organization-activity-organization" x-model="selectedOrganization">
						<option
							value="0"><?php esc_html_e( 'Select an organization', 'ld-organization' ); ?></option>
						<?php foreach ( $all_organizations as $organization ) : ?>
							<option
								value="<?php echo esc_attr( $organization->ID ); ?>"><?php echo esc_html( $organization->post_title ); ?></option>
						<?php endforeach; ?>
					</select>
					<span x-show="loading" x-cloak x-transition class="loading-icon">
						<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-loader-icon lucide-loader"><path d="M12 2v4"/><path d="m16.2 7.8 2.9-2.9"/><path d="M18 12h4"/><path d="m16.2 16.2 2.9 2.9"/><path d="M12 18v4"/><path d="m4.9 19.1 2.9-2.9"/><path d="M2 12h4"/><path d="m4.9 4.9 2.9 2.9"/></svg>
						<span class="sr-only">
							<?php esc_html_e( 'Loading...', 'ld-organization' ); ?>
						</span>
					</span>
					<select id="ld-organization-activity-course" x-model="selectedCourse" x-show="!loading && courses.length > 0" x-cloak x-transition>
						<option value="0"><?php esc_html_e( 'Select a course', 'ld-organization' ); ?></option>
						<template x-for="course in courses" :key="course.id">
							<option :value="course.id" x-text="course.title"></option>
						</template>
					</select>
				</div>
			</div>
			<div class="main-table-wrapper" 
				id="ld-organization-activity-table"
				x-show="selectedOrganization !== '0' && selectedCourse !== '0'"
				x-cloak
				x-transition
				:data-organization-id="selectedOrganization"
				:data-course-id="selectedCourse"
			>
				<table id="ld-organization-activity"></table>
				<script>
					document.addEventListener('alpine:init', () => {
						Alpine.store('table', {
							table: null,
							organizationId: null,
							courseId: null,
							renderTable(organizationId, courseId) {
								this.organizationId = organizationId;
								this.courseId = courseId;
								if (this.table) {
									this.table.ajax.reload();
								} else {
									this.table = new DataTable('#ld-organization-activity', {
										language: {
											url: '//cdn.datatables.net/plug-ins/2.2.2/i18n/fi.json',
										},
										layout: {
											topStart: {
												buttons: [
													{
														extend: 'pageLength',
													},
													{
														extend: 'searchPanes',
													},
													{
														extend: 'selected',
														text: '<?php esc_html_e( 'Reset', 'ld-organization' ); ?>',
														action: async function ( e, dt, node, config ) {
															dt.processing(true);
															let rows = dt.rows( { selected: true } );
															let userIds = Array.from(rows.data()).map(row => row.user_id);
															const response = await fetch('<?php echo esc_url( admin_url( 'admin-ajax.php' ) ); ?>', {
																method: 'POST',
																body: new URLSearchParams({
																	action: 'reset_org_course_activity',
																	organization_id: jQuery('#ld-organization-activity-table').data('organization-id'),
																	course_id: jQuery('#ld-organization-activity-table').data('course-id'),	
																	user_ids: userIds,
																	nonce: '<?php echo esc_js( wp_create_nonce( 'ld-organization-course-reset-activity-nonce' ) ); ?>',
																}),
															});
															dt.processing(false);
															if (!response.ok) {
																new Notyf().error('<?php esc_html_e( 'An error occurred', 'ld-organization' ); ?>');
																return;
															}
															const responseData = await response.json();
															if (responseData.success) {
																Alpine.store('table').table.ajax.reload();
																new Notyf().success('<?php esc_html_e( 'Course activity reset successfully', 'ld-organization' ); ?>');
															} else {
																if (responseData.data.error === 'no_quizzes_found') {
																const warn = new Notyf({
																	types: [
																		{
																			type: 'warning',
																			background: '#ffb347',
																			icon: false,
																		}
																	]
																});
																warn.open({
																	type: 'warning',
																	message: '<?php esc_html_e( 'No quizzes found for this course', 'ld-organization' ); ?>',
																})
																} else {
																	new Notyf().error('<?php esc_html_e( 'An error occurred', 'ld-organization' ); ?>');
																}
															}
														}
													},
												],
											}
										},
										processing: true,
										ajax: {
											url: '<?php echo esc_url( admin_url( 'admin-ajax.php' ) ); ?>',
											type: 'POST',
											data: function (data) {
												return {
													action: 'fetch_org_course_activity',
													organization_id: this.organizationId,
													course_id: this.courseId,
													nonce: '<?php echo esc_js( wp_create_nonce( 'ld-organization-course-activity-nonce' ) ); ?>',
												};
											}.bind(this),
											dataSrc: function(json) {
												if (json.success) {
													return json.data.users;
												}
												return [];
											},
										},
										select: {
											style: 'multi+shift',

										},
										createdRow: function(row, data) {
											if (data.is_threshold) {
												jQuery(row).addClass('threshold');
											}
											if (data.reset_date !== '-') {
												jQuery(row).addClass('has-reset-date');
											}
										},
										columns: [
											{ render: DataTable.render.select(), sortable: false, searchable: false },
											{ data: 'user_id', visible: false },
											{ data: 'user_name', type: 'string-utf8', title: '<?php esc_html_e( 'User name', 'ld-organization' ); ?>' },
											{ data: 'user_email', type: 'string-utf8', title: '<?php esc_html_e( 'User email', 'ld-organization' ); ?>' },
											{ 
												data: 'completed', 
												title: '<?php esc_html_e( 'Completed', 'ld-organization' ); ?>',
												type: 'html',
												render: function(val) {
													return (val === 'true' || val === true ? '<input type="checkbox" checked disabled><span class="sr-only">true</span>' : '<input type="checkbox" disabled><span class="sr-only">false</span>');
												}
											},
											{ 
												data: 'completed_date', 
												title: '<?php esc_html_e( 'Completed date', 'ld-organization' ); ?>',
											},
											{ 
												data: 'is_threshold', 
												title: '<?php esc_html_e( 'Expiring soon', 'ld-organization' ); ?>',
												type: 'html',
												render: function(val) {
													return (val === 'true' || val === true ? '<input type="checkbox" checked disabled><span class="sr-only">true</span>' : '<input type="checkbox" disabled><span class="sr-only">false</span>');
												},
											},
											{ data: 'reset_date', type: 'string-utf8', title: '<?php esc_html_e( 'Reset date', 'ld-organization' ); ?>' },
										],
									});
								}
							}
						})
					});
				</script>
			</div>
		</div>
		<style>
			.wppb-serial-notification {
				display: none;
			}

			.select-wrapper {
				display: flex;
				flex-direction: column;
				justify-content: center;
				padding: 1rem 0;
			}
			.sr-only {
				position: absolute;
				left: -10000px;
				top: auto;
				width: 1px;
				height: 1px;
			}

			.select-data-wrapper {
				padding: 0.5rem 0;
				display: flex;
				flex-direction: row;
				justify-content: flex-start;
				align-items: center;
				gap: 0.5rem;
				flex-wrap: wrap;
			}

			.loading-icon svg {
				animation: spin 1s linear infinite;
			}

			@keyframes spin {
				0% { transform: rotate(0deg); }
				100% { transform: rotate(360deg); }
			}

			#ld-organization-activity-course, #ld-organization-activity-organization {
				max-width: 200px;
			}

			[x-cloak] {
				display: none;
			}

			.main-table-wrapper .dt-select-checkbox {
				background: unset;
				min-width: unset;
				margin: unset !important;
				margin-top: 1px !important;
			}

			.main-table-wrapper .dt-select-checkbox:checked::before {
				display: none;
			}

			tr.threshold {
				background-color: #ffb347 !important;
			}

			tr.has-reset-date {
				background-color: #48aa43 !important;
				color: #fff !important;
			}

			td input[type="checkbox"]:disabled {
				opacity: 1 !important;
				background-color: #fff !important;
			}

			tr.threshold.selected td {
				box-shadow: inset 0 0 0 9999px rgb(253, 148, 0) !important;
			}

			tr.has-reset-date.selected td {
				box-shadow: inset 0 0 0 9999px rgb(74, 195, 67) !important;
			}
		</style>
		<?php
	}
}
