<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
/**
 * Organization class
 *
 * @package LD_Organization
 */

namespace LD_Organization;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use Exception;
use InvalidArgumentException;
use LD_Organization\Exceptions\OrganizationNotFoundException;
use LD_Organization\Exceptions\OrganizationOutOfLicensesException;
use LD_Organization\Exceptions\PermissionException;
use WP_Post;
use WP_User;
use function Sentry\captureException;

/**
 * The Organization class is responsible fetching and checking different attributes revolving around organizations,
 * can and should be used when directly communicating with Learndash.
 *
 * @package LD_Organization
 * @since 0.1.0
 */
class Organization {
	public const ORGANIZATION_ROLES            = array( 'organization_owners', 'group_leaders' );
	public const ALLOWED_ROLE_ARGS             = array( 'both', 'organization_owners', 'group_leaders' );
	public const WP_ROLE_NAME_OWNERS           = array( 'organisaatio', 'administrator' );
	public const WP_ROLE_NAME_GROUP_LEADERS    = array( 'organisaatio', 'group_leader', 'opettaja', 'administrator' );
	public const ORGANIZATION_COMPLETIONS_META = '_organization_completions';

	/**
	 * The post type slug to use.
	 * Default: "organization".
	 *
	 * @var string
	 */
	public static string $post_type = POST_TYPE_NAME;

	/**
	 * The organization being handled.
	 *
	 * @var WP_Post|null
	 */
	private ?WP_Post $organization;

	/**
	 * All available transients for this class.
	 *
	 * @var array|string[]
	 */
	public static array $transients = array(
		'allowed_courses'     => 'allowed_courses',
		'organization_groups' => 'organization_groups',
	);

	/**
	 * TODO: Needs a constructor?
	 */
	public function __construct() {
	}

	/**
	 * Handles initializing Organization class with ID.
	 *
	 * @param int $id The ID of the organization to retrieve.
	 *
	 * @return Organization
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws InvalidArgumentException If the ID belongs to a non-organization post.
	 */
	public static function with_id( int $id ): Organization {
		$instance = new self();
		$post     = $instance->get_post_by_id( $id );
		$instance->set_organization( $post );
		return $instance;
	}

	/**
	 * Handles initializing Organization class based on a user ID.
	 *
	 * @param int    $id The ID of a user belonging to the organization to retrieve.
	 * @param string $key The meta key to query on, uses Organization::get_post_by_user_id so the accepted values are the same as that.
	 *
	 * @return Organization
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws InvalidArgumentException If the ID belongs to a non-organization post or the passed key is invalid.
	 *
	 * @uses Organization::get_post_by_user_id()
	 */
	public static function by_user_id( int $id, string $key = 'both' ): Organization {
		$instance = new self();
		$post     = $instance->get_post_by_user_id( $id, $key );
		$instance->set_organization( $post );
		return $instance;
	}

	/**
	 * Handles initializing Organization class based on a user ID.
	 *
	 * @param string $email The email of the user belonging to an organization.
	 * @param string $key The meta key to query on, uses Organization::get_post_by_user_id so the accepted values are the same as that.
	 *
	 * @return Organization
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws InvalidArgumentException If the ID belongs to a non-organization post or the passed key is invalid.
	 *
	 * @uses Organization::get_post_by_user_email()
	 */
	public static function by_user_email( string $email, string $key = 'both' ): Organization {
		$instance = new self();
		$post     = $instance->get_post_by_user_email( $email, $key );
		$instance->set_organization( $post );
		return $instance;
	}

	/**
	 * Handles getting the associated organization with a certain group ID.
	 *
	 * @param int $group_id The group ID to get organization from.
	 *
	 * @return Organization
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws InvalidArgumentException If the ID is not a valid group ID.
	 */
	public static function by_group_id( int $group_id ): Organization {
		$instance = new self();
		$post     = $instance->get_by_group_id( $group_id );
		$instance->set_organization( $post );
		return $instance;
	}

	/**
	 * Handles actually fetching the organization from the Group ID.
	 *
	 * @param int $group_id The ID of the group to fetch from.
	 *
	 * @return WP_Post
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws InvalidArgumentException If the ID is not a valid group ID.
	 */
	private function get_by_group_id( int $group_id ): WP_Post {
		if ( ! learndash_is_group_post( $group_id ) ) {
			throw new InvalidArgumentException( __( 'The provided ID is not a valid group ID.', 'ld-organization' ) );
		}
		$organization_id = get_field( 'group_organization', $group_id );
		return $this->get_post_by_id( $organization_id );
	}

	/**
	 * Handles initializing Organization class based on the user ID.
	 *
	 * @param string $email The email to lookup user by.
	 * @param string $key The meta key to query on, uses Organization::get_post_by_user_id so the accepted values are the same as that.
	 *
	 * @return WP_Post
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws InvalidArgumentException If a user is not found, because the email is not found.
	 *
	 * @uses Organization::get_post_by_user_id()
	 */
	private function get_post_by_user_email( string $email, string $key = 'both' ): WP_Post {
		$user = get_user_by( 'email', $email );
		if ( ! $user ) {
			throw new InvalidArgumentException( 'No user with the provided email found' );
		}
		return $this->get_post_by_user_id( $user->ID, $key );
	}

	/**
	 * Handles fetching the post by ID.
	 *
	 * @param int|null $id The Post id.
	 *
	 * @return WP_Post
	 * @throws OrganizationNotFoundException Throws this exception if a Organization cannot be found.
	 */
	private function get_post_by_id( int|null $id ): WP_Post {
		$post = get_post( $id );
		if ( ! $post ) {
			throw new OrganizationNotFoundException();
		}
		if ( self::$post_type !== $post->post_type ) {
			throw new InvalidArgumentException( __( 'ID does not belong to a Organization Post', 'ld-organization' ) );
		}
		return $post;
	}

	/**
	 * Handles fetching the Organization based on the user ID.
	 *
	 * @param int    $id The user ID to search for.
	 * @param string $key The key to query for, defaults to 'organization_owners', can be either 'organization_owners', 'group_leaders' or 'both'.
	 *
	 * @return WP_Post
	 * @throws InvalidArgumentException If the key is not in the allowed ranges.
	 * @throws OrganizationNotFoundException Throws this exception if no organization is found.
	 */
	private function get_post_by_user_id( int $id, string $key = 'both' ): WP_Post {
		$args = array(
			'post_type'   => self::$post_type,
			'numberposts' => 1,
		);
		if ( 'both' === $key ) {
			$args['meta_query'] = array(
				'relation' => 'OR',
				array(
					'key'     => 'organization_owners',
					'value'   => '"' . $id . '"',
					'compare' => 'LIKE',
				),
				array(
					'key'     => 'group_leaders',
					'value'   => '"' . $id . '"',
					'compare' => 'LIKE',
				),
			);
		} elseif ( in_array( $key, array( 'organization_owners', 'group_leaders' ), true ) ) {
			$args['meta_query'] = array(
				array(
					'key'     => $key,
					'value'   => '"' . $id . '"',
					'compare' => 'LIKE',
				),
			);
		} else {
			throw new InvalidArgumentException( "The key needs to be either: 'organization_owners', 'group_leaders' or 'both'" );
		}
		$posts = get_posts(
			$args
		);

		if ( empty( $posts ) ) {
			throw new OrganizationNotFoundException();
		}
		return $posts[0];
	}

	/**
	 * Fetch transient name based on key.
	 *
	 * @param string $key The key to fetch.
	 * @param int    $id The ID to make the transient name unique.
	 *
	 * @return false|string
	 */
	public static function get_transient_name( string $key, int $id ): string {
		return isset( self::$transients[ $key ] ) ? self::$transients[ $key ] . '_' . $id : false;
	}

	/**
	 * Set the used licenses to a fixed number
	 *
	 * This method does a bunch of checks to see if we are allowed to do this.
	 *
	 * @param int $used_licenses The number of licenses to set as used.
	 *
	 * @return int|bool
	 * @throws OrganizationNotFoundException In case there is no organization defined on the object.
	 * @throws InvalidArgumentException Will be thrown if the number of licenses exceeds the allowed licenses on the organization.
	 */
	final public function set_used_licenses( int $used_licenses ): int {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		$allowed_licenses = $this->get_licenses();
		if ( $used_licenses > $allowed_licenses ) {
			throw new InvalidArgumentException( __( 'Used licenses cannot be higher than available licenses', 'ld-organization' ) );
		}
		update_post_meta( $this->organization->ID, USED_LICENSES_META_KEY, $used_licenses );

		return $used_licenses;
	}

	/**
	 * Handles adding the user to the organization.
	 *
	 * @param int    $user_id The user id to add.
	 * @param string $role The role of user to add.
	 *
	 * @return void
	 * @throws OrganizationNotFoundException Thrown when the organization does not exist on the object.
	 * @throws PermissionException Thrown when the calling user does not have the correct permissions.
	 * @throws InvalidArgumentException When the user is already part of the organization.
	 *
	 * @since 0.8.2
	 */
	final public function add_user_to_organization( int $user_id, string $role ): void {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		if ( ! in_array( $role, self::ORGANIZATION_ROLES, true ) ) {
			throw new InvalidArgumentException( 'Passed role is not allowed' );
		}

		$current_user_id = get_current_user_id();

		if ( ! $this->has_user( $current_user_id, $role ) ) {
			throw new PermissionException( 'User does not have permission to add a user to this organization' );
		}

		if ( $this->has_user( $user_id ) ) {
			throw new InvalidArgumentException( 'User is already part of organization' );
		}

		$new_user = get_user_by( 'ID', $user_id );
		if ( empty( $new_user ) ) {
			throw new InvalidArgumentException( 'Provided User ID is not valid' );
		}

		$current_users = $this->get_group_leaders();
		$new_users     = array_merge( $current_users, array( $new_user ) );

		update_field( $role, $new_users, $this->organization->ID );

		/**
		 * Fires upon successful addition of a user to an organization.
		 *
		 * @param int $organization_id ID of organization.
		 * @param int $user_id The user added to the organization
		 * @param string $role The role of the user
		 *
		 * @since 0.12.0
		 */
		do_action( 'ldorg_organization_user_added', $this->organization->ID, $user_id, $role );
	}

	/**
	 * Increases the used licenses by the specified amount
	 *
	 * This method does a bunch of checks to see if we are allowed to do this.
	 *
	 * @param int $amount The amount of licenses to "use".
	 *
	 * @return int|bool
	 * @throws OrganizationNotFoundException In case there is no organization defined on the object.
	 * @throws InvalidArgumentException If the amount would make the used_licenses larger than the maximum allowed licenses.
	 */
	final public function increment_used_licenses( int $amount ): int {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		$allowed_licenses      = $this->get_licenses();
		$current_used_licenses = $this->get_used_licenses();
		if ( ( $amount + $current_used_licenses ) > $allowed_licenses ) {
			throw new InvalidArgumentException( __( 'Used licenses cannot be higher than available licenses', 'ld-organization' ) );
		}
		update_post_meta( $this->organization->ID, USED_LICENSES_META_KEY, ( $amount + $current_used_licenses ) );

		return ( $amount + $current_used_licenses );
	}

	/**
	 * Creates a group and handles assigning the group_leaders
	 *
	 * @param int   $group_id The group ID.
	 * @param int[] $group_leaders An array of initial group leaders.
	 *
	 * @return bool
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws OrganizationOutOfLicensesException If the organization does not have enough licenses left.
	 * @throws InvalidArgumentException If the passed group ID is invalid.
	 *
	 * @since 0.4.0
	 */
	final public function initialize_group( int $group_id, array $group_leaders ): bool {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		if ( ! $this->is_school() && $this->get_remaining_licenses() <= 0 ) {
			throw new OrganizationOutOfLicensesException();
		}

		if ( ! learndash_is_group_post( $group_id ) ) {
			throw new InvalidArgumentException( 'Group ID is not a valid group' );
		}

		learndash_set_groups_administrators( $group_id, $group_leaders );
		$result = update_field( 'group_organization', $this->organization->ID, $group_id );

		/**
		 * Calls the filter ldorg_should_recalculate_on_creation
		 *
		 * Should return true if we should increment the used licenses.
		 *
		 * @since 0.4.0
		 *
		 * @param bool $should_increment Whether to increment the used licenses on creation or not.
		 * @param WP_Post $organization The current organization post.
		 * @param int $group_id The created group ID.
		 */
		if ( $result && ! $this->is_school() && apply_filters( 'ldorg_should_recalculate_on_creation', true, $this->organization, $group_id ) ) {
			$this->recalculate_licenses();
		}

		return $result;
	}


	/**
	 * Handles recalculating the amount of used licenses
	 *
	 * @return void
	 *
	 * @since 0.12.0 Added recalculate_licenses method to recalculate the amount of used licenses.
	 * @throws OrganizationNotFoundException If the organization is not found on the current object.
	 * @throws InvalidArgumentException If the Organization does not have enough licenses for this.
	 */
	final public function recalculate_licenses(): void {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		if ( $this->is_school() ) {
			return;
		}

		$organization_groups = $this->get_organization_groups();

		$users_list = array();

		foreach ( $organization_groups as $group_id ) {
			$group_users    = learndash_get_groups_user_ids( $group_id, true );
			$administrators = learndash_get_groups_administrator_ids( $group_id, true );

			// Remove all the users that are not only members but also administrators.
			$only_users = array_diff( $group_users, $administrators );

			$users_list = array( ...$users_list, ...$only_users );
		}

		$final_list = array_unique( $users_list, SORT_NUMERIC );

		$this->set_used_licenses( count( $final_list ) );
	}

	/**
	 * Checks if the user is part of the owners of the organization.
	 *
	 * @param int    $user_id The user ID to check against.
	 * @param string $role The role to check against, can be either 'organization_owners' or 'group_leaders' or 'both'.
	 * @param bool   $strict Whether to check exactly for the requested role or to allow "owners" to pass.
	 *
	 * @return bool
	 * @throws OrganizationNotFoundException If this object has no organization is defined.
	 * @throws InvalidArgumentException If the role is not a valid value.
	 */
	final public function has_user( int $user_id, string $role = 'both', bool $strict = false ): bool {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		if ( ! in_array( $role, self::ALLOWED_ROLE_ARGS, true ) ) {
			throw new InvalidArgumentException( "Role has to be either 'organization_owners' or 'group_leaders' or 'both'" );
		}

		// Fetch both if that is required and merge the arrays.
		if ( 'both' === $role ) {
			$owners        = get_field( 'organization_owners', $this->organization->ID );
			$group_leaders = get_field( 'group_leaders', $this->organization->ID );
			$users         = array_merge( $owners, $group_leaders );
		} else {
			$users = get_field( $role, $this->organization->ID );
		}

		if ( ! is_array( $users ) || empty( $users ) ) {
			return false;
		}

		$filtered_arr = array_filter(
			$users,
			static function ( WP_User $user ) use ( $user_id ) {
				return $user_id === $user->ID;
			}
		);

		// If we are checking for group_leaders, also check if the user is an owner and allow access.
		if ( ! $strict && 'group_leaders' === $role && empty( $filtered_arr ) ) {
			$owners      = get_field( 'organization_owners', $this->organization->ID );
			$check_owner = array_filter(
				$owners,
				static function ( WP_User $user ) use ( $user_id ) {
					return $user_id === $user->ID;
				}
			);
			return ! empty( $check_owner );
		}

		return ! empty( $filtered_arr );
	}

	/**
	 * Fetches the number of used licenses from the database.
	 *
	 * @param bool $skip_recalc Should the recalculate be skipped.
	 *
	 * @return int
	 * @throws OrganizationNotFoundException If the organization is not defined on the object.
	 */
	final public function get_used_licenses( bool $skip_recalc = false ): int {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		if ( ! $skip_recalc ) {
			$this->recalculate_licenses();
		}

		$number_used_licenses = (int) get_post_meta( $this->organization->ID, USED_LICENSES_META_KEY, true );
		if ( empty( $number_used_licenses ) && 0 !== $number_used_licenses ) {
			return 0;
		}

		return $number_used_licenses;
	}

	/**
	 * Handles fetching the amount of allowed licenses for this organization.
	 *
	 * @return int
	 * @throws OrganizationNotFoundException If the organization is not defined on the object.
	 */
	final public function get_licenses(): int {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		return (int) get_field( 'licenses', $this->organization->ID, true );
	}

	/**
	 * Returns the remaining licenses of this organization.
	 *
	 * @param bool $skip_recalc Should we skip the recalculation of licenses.
	 *
	 * @return int
	 * @throws OrganizationNotFoundException Thrown if the organization is not defined on the object.
	 * @since 0.12.1 Return correct amount for school organizations.
	 * @since 0.17.2 Added skip_recalc optional parameter.
	 */
	final public function get_remaining_licenses( bool $skip_recalc = false ): int {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		if ( $this->is_school() ) {
			return 999999;
		}

		if ( ! $skip_recalc ) {
			// We recalculate the licenses when checking.
			$this->recalculate_licenses();
		}

		return $this->get_licenses() - $this->get_used_licenses();
	}

	/**
	 * Handles returning a list of allowed course ids for this organization.
	 *
	 * @param bool $force Should the transient be cleared.
	 *
	 * @return int[]
	 * @throws OrganizationNotFoundException If the organization does not exist on the object.
	 */
	final public function get_allowed_courses_ids( bool $force = false ): array {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		// Returns a transient if found and force is not true.
		$transient_name = self::get_transient_name( 'allowed_courses', $this->organization->ID );
		if ( ! $force && false !== $transient_name && ( false !== ( $value = TransientManager::get( $transient_name ) ) ) ) {
			return $value;
		}
		if ( ! $this->is_school() ) {
			$courses = get_all_course_ids();
			if ( ! empty( $courses ) ) {
				TransientManager::set( $transient_name, $courses, HOUR_IN_SECONDS );
			}
			return $courses;
		}

		$tags = $this->get_allowed_tags();
		if ( empty( $tags ) ) {
			return array();
		}
		$course_slug = learndash_get_post_type_slug( 'course' );

		// Builds an array of arguments, we use the custom post type slug and build a taxonomy query for the ld_course_tag.
		$args = array(
			'numberposts' => -1,
			'post_type'   => $course_slug,
			'fields'      => 'ids',
			'tax_query'   => array(
				array(
					'taxonomy' => 'ld_course_tag',
					'terms'    => $tags,
					'field'    => 'term_id',
				),
			),
		);

		$courses = get_posts( $args );
		if ( ! empty( $courses ) && false !== $transient_name ) {
			TransientManager::set( $transient_name, $courses, HOUR_IN_SECONDS );
		}

		return $courses;
	}

	/**
	 * Handles adding the post meta for completions to the organization.
	 *
	 * @param int|string $year The year to add the completion to.
	 * @param array      $completion {
	 *                  The completion array.
	 *
	 * @type int $course_id The course ID.
	 * @type int|WP_User $user_id The user ID.
	 * }
	 *
	 * @return void
	 * @throws OrganizationNotFoundException If the organization does not exist on the object.
	 */
	final public function add_completion( int|string $year, array $completion ): void {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		if ( empty( $completion['course_id'] ) ) {
			return;
		}
		$year                = (string) $year;
		$current_completions = get_post_meta( $this->organization->ID, self::ORGANIZATION_COMPLETIONS_META, true );
		if ( empty( $current_completions ) || ! is_array( $current_completions ) ) {
			$current_completions = array();
		}
		if ( ! isset( $current_completions[ $year ] ) ) {
			$current_completions[ $year ] = array(
				'courses' => array(
					$completion['course_id'] => 0,
				),
			);
		}
		if ( ! isset( $current_completions[ $year ]['courses'][ $completion['course_id'] ] ) ) {
			$current_completions[ $year ]['courses'][ $completion['course_id'] ] = 0;
		}
		++$current_completions[ $year ]['courses'][ $completion['course_id'] ];
		$updated = update_post_meta( $this->organization->ID, self::ORGANIZATION_COMPLETIONS_META, $current_completions );
		if ( ! $updated ) {
			captureException( new Exception( 'Could not update post meta for organization completions' . $this->organization->ID, 0 ) );
		}
	}

	/**
	 * Handles fetching the completions for a certain year.
	 *
	 * @param int|string $year The year to fetch completions for.
	 *
	 * @return array
	 * @throws OrganizationNotFoundException If the organization does not exist on the object.
	 */
	final public function get_completions( int|string $year ): array {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		$year                = (string) $year;
		$current_completions = get_post_meta( $this->organization->ID, self::ORGANIZATION_COMPLETIONS_META, true );
		$defaults            = array(
			'courses' => array(),
		);
		if ( empty( $current_completions ) || ! is_array( $current_completions ) ) {
			return $defaults;
		}

		return $current_completions[ $year ] ?? $defaults;
	}

	/**
	 * Gets all completions for the organization.
	 *
	 * @return array
	 * @throws OrganizationNotFoundException If the organization does not exist on the object.
	 */
	final public function get_all_completions(): array {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		$current_completions = get_post_meta( $this->organization->ID, self::ORGANIZATION_COMPLETIONS_META, true );
		if ( empty( $current_completions ) || ! is_array( $current_completions ) ) {
			return array();
		}

		return $current_completions;
	}
	/**
	 * Returns whether this class has an organization or not.
	 *
	 * @return bool
	 */
	final public function has_organization(): bool {
		return isset( $this->organization );
	}

	/**
	 * Sets the organization.
	 *
	 * @param WP_Post|null $organization The organization or null.
	 */
	final public function set_organization( ?WP_Post $organization ): void {
		$this->organization = $organization;
	}

	/**
	 * Returns the organization WP_Post object.
	 *
	 * @return WP_Post|null
	 */
	final public function get_organization(): ?WP_Post {
		return $this->organization;
	}

	/**
	 * Returns the allowed tags for this organization.
	 *
	 * @return array|int
	 * @throws OrganizationNotFoundException If the class has no organizations.
	 * @throws InvalidArgumentException If the method is called on a school.
	 *
	 * @since 0.5.0
	 */
	final public function get_allowed_tags() {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		if ( ! $this->is_school() ) {
			throw new InvalidArgumentException( 'This method should only be called on schools.' );
		}
		return get_field( 'allowed_course_tags', $this->organization->ID );
	}

	/**
	 * Checks if the requested user is in any of the organizations groups.
	 *
	 * @param int $user_id The user ID.
	 *
	 * @return bool
	 * @throws InvalidArgumentException If the requested user ID is invalid.
	 * @throws OrganizationNotFoundException Thrown if the organization does not exists on this object.
	 */
	final public function is_user_in_organization_groups( int $user_id ): bool {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		if ( ! get_user_by( 'ID', $user_id ) ) {
			throw new InvalidArgumentException( 'User ID is invalid' );
		}

		$group_ids = $this->get_organization_groups();

		$users_group_ids = learndash_get_users_group_ids( $user_id, true );
		$intersect       = array_intersect( $group_ids, $users_group_ids );
		return ! empty( $intersect );
	}

	/**
	 * Fetches the organizations groups.
	 *
	 * @return int[]
	 * @throws OrganizationNotFoundException If the requested organization does not exist on this object.
	 */
	final public function get_organization_groups(): array {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		$transient_name = self::get_transient_name( 'organization_groups', $this->organization->ID );

		if ( false !== ( $value = TransientManager::get( $transient_name ) ) ) {
			return $value;
		}

		$groups = get_posts(
			array(
				'post_type'   => learndash_get_post_type_slug( 'group' ),
				'meta_query'  => array(
					array(
						'key'   => 'group_organization',
						'value' => $this->organization->ID,
					),
				),
				'numberposts' => -1,
				'fields'      => 'ids',
			)
		);

		if ( ! empty( $groups ) ) {
			TransientManager::set( $transient_name, $groups, 5 * MINUTE_IN_SECONDS );
		}
		return $groups;
	}

	/**
	 * Returns the organizations type as a string.
	 *
	 * @return string
	 * @throws OrganizationNotFoundException Thrown if the organization does not exists on this object.
	 */
	final public function get_type(): string {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		} elseif ( get_field( 'organization_type', $this->organization->ID, true ) === null ) {
			throw new OrganizationNotFoundException();
		}

		return get_field( 'organization_type', $this->organization->ID, true );
	}

	/**
	 * Checks whether this organization is of type school.
	 *
	 * @return bool
	 * @throws OrganizationNotFoundException Thrown if the organization does not exists on this object.
	 * @uses Organization::get_type()
	 * @since 0.3.0
	 */
	final public function is_school(): bool {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		return $this->get_type() === 'school';
	}

	/**
	 * Returns a list of WP_Users.
	 *
	 * @return WP_User[]
	 * @throws OrganizationNotFoundException Thrown if the organization does not exists on this object.
	 * @since 0.8.2
	 */
	final public function get_group_leaders(): array {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		return get_field( 'group_leaders', $this->organization->ID );
	}
}
