<?php
/**
 * Handles registering the RestAPI endpoints for handling organizations.
 *
 * @package LD_Organization
 */

namespace LD_Organization;

use InvalidArgumentException;
use LDLMS_Model_Course;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * RestAPI class, when initialized will register the restAPI.
 *
 * @package LD_Organization
 * @since 0.2.0
 */
class RestAPI {

	/**
	 * The RestAPI namespace.
	 *
	 * @var string
	 */
	private string $namespace = 'ld-organization';

	/**
	 * The RestAPI version.
	 *
	 * @var string
	 */
	private string $version = 'v1';

	/**
	 * Registers all required functionality and endpoints.
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_endpoints' ) );
	}

	/**
	 * Registers all the endpoint and their parameters.
	 *
	 * @return void
	 */
	final public function register_endpoints(): void {
		$this->register_organization_routes();
		$this->register_groups_routes();
		$this->register_user_routes();
	}

	/**
	 * Gets data of the courses in requested organization
	 */
	final public function get_organization_courses( WP_REST_Request $request): WP_REST_Response {
		try {
			$org_id = $request->get_param( 'id' );
			$organization = organization::with_id($org_id);

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$transient_name = 'org_'. $org_id .'_course_data';

			$data = array();

			$groups = $organization->get_organization_groups();

			foreach ($groups as $group_id) {

				if( ! $group_id ){
					continue;
				}

				$courses = learndash_get_group_courses_list($group_id);

				foreach ($courses as $course_id) {

					if ( ! $course_id ){
						continue;
					}

					$course = get_post($course_id);

					// Excludes admins
					$enrolled_users_query = learndash_get_users_for_course( $course_id, array(), true );

					$enrolled_users_id = $enrolled_users_query->get_results();

					$enrolled_users = count($enrolled_users_id);

					$course_completions = 0;

					foreach ($enrolled_users_id as $enrolled_user_id) {
						$progress = learndash_user_get_course_progress($enrolled_user_id, $course_id);
						if ($progress["completed"] === $progress["total"]) {
							++ $course_completions;
						}
					}

					$data[] = array(
						'course_id' => $course_id,
						'title' => $course->post_title,
						'enrolled' => $enrolled_users,
						'enrolled_id' => $enrolled_users_id,
						'completions' => $course_completions,
					);

				}

			}

			if ( ! empty( $data ) ) {
				//LDOrgTransientManager::set( $transient_name, $data );
			}

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $data,
					'message' => "Successfully fetched courses for organization ".$org_id
				)
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'Organization not found', 'ld-organization' ),
				),
				404
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Gets the requested users courses and the progress for the requested group.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_user_group_progress( WP_REST_Request $request ): WP_REST_Response {
		try {
			$group_id          = $request->get_param( 'group_id' );
			$requested_user_id = $request->get_param( 'user_id' );
			$current_user_id   = get_current_user_id();
			$organization      = Organization::by_user_id( $current_user_id );

			if ( ! $organization->is_user_in_organization_groups( $requested_user_id ) ) {
				return new WP_REST_Response(
					array(
						'success' => false,
						'message' => __( 'Requested user is not part of the current users organization', 'ld-organization' ),
					),
					403
				);
			}

			$transient_name = 'users_group_' . $group_id . '_course_' . $requested_user_id;

			if ( false !== ( $value = LDOrgTransientManager::get( $transient_name ) ) ) {
				return new WP_REST_Response(
					array(
						'success' => true,
						'data'    => $value,
					)
				);
			}

			$courses = learndash_get_group_courses_list( $group_id );

			$data = array();

			// Loops over each course
			foreach ( $courses as $course_id ) {
				$course          = get_post( $course_id );
				$course_progress = learndash_user_get_course_progress( $requested_user_id, $course_id );

				$lessons      = learndash_get_course_lessons_list( $course_id, $requested_user_id );
				$lessons_data = array();

				// Loops over each lesson.
				foreach ( $lessons as $lesson ) {

					if ( ! $lesson ) {
						continue;
					}

					$lesson_data = $lesson['post'];

					// Gets topics of lesson.
					$topics                    = learndash_course_get_topics( $course_id, $lesson_data->ID );
					$topics_data               = array();
					$lesson_progress_topics    = $course_progress['topics'][ $lesson_data->ID ];
					$lesson_progress_total     = count( $lesson_progress_topics );
					$lesson_progress_completed = array_reduce(
						array_keys( $lesson_progress_topics ),
						static function ( $carry, $key ) use ( $lesson_progress_topics ) {
							return $carry + $lesson_progress_topics[ $key ];
						}
					);

					// Calculate percentage of lesson progress.
					$progress_percentage_lesson = 0;
					if ( $lesson_progress_completed !== 0 && $lesson_progress_total !== 0 && $lesson_progress_completed <= $lesson_progress_total ) {
						$progress_percentage_lesson = handle_erroneous_values( ( $lesson_progress_completed / $lesson_progress_total ) * 100 );
						$progress_percentage_lesson = round( $progress_percentage_lesson );
					}

					foreach ( $topics as $topic ) {
						if ( ! $topic ) {
							continue;
						}

						$topic_completed = 1 === $lesson_progress_topics[ $topic->ID ];

						$topics_data[] = array(
							'title'     => $topic->post_title,
							'id'        => $topic->ID,
							'completed' => $topic_completed,
						);

					}

					$lessons_data[] = array(
						'id'                            => $lesson_data->ID,
						'title'                         => $lesson_data->post_title,
						'topics'                        => $topics_data,
						'progress_completed_percentage' => $progress_percentage_lesson,
					);
				}

				$quizzes      = learndash_get_course_quiz_list( $course_id );
				$quizzes_data = array();

				// Loops over each quiz.
				foreach ( $quizzes as $quiz ) {

					if ( ! $quiz ) {
						continue;
					}

					$quiz_data      = $quiz['post'];
					$quiz_completed = learndash_user_quiz_has_completed( $requested_user_id, $quiz_data->ID, $course_id );

					$quizzes_data[] = array(
						'id'        => $quiz_data->ID,
						'title'     => $quiz_data->post_title,
						'completed' => $quiz_completed,
					);

				}

				if ( ! $course ) {
					continue;
				}

				// Calculate percentage of course progress.
				$progress_percentage = 0;
				if ( $course_progress['completed'] !== 0 && $course_progress['total'] !== 0 && $course_progress['completed'] <= $course_progress['total'] ) {
					$progress_percentage = handle_erroneous_values( ( $course_progress['completed'] / $course_progress['total'] ) * 100 );
					$progress_percentage = round( $progress_percentage );
				}

				$data[] = array(
					'id'                            => $course->ID,
					'title'                         => $course->post_title,
					'progress_completed_percentage' => $progress_percentage,
					'lessons'                       => $lessons_data,
					'quizzes'                       => $quizzes_data,
				);
			}

			if ( ! empty( $data ) ) {
				LDOrgTransientManager::set( $transient_name, $data );
			}

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $data,
				)
			);
		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'Organization not found', 'ld-organization' ),
				),
				404
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	final public function get_user_certificates( WP_REST_Request $request): WP_REST_Response{

		$response = null;

		try {
			$requested_user_id = $request->get_param( 'user_id' );
			$current_user_id   = get_current_user_id();
			$organization      = Organization::by_user_id( $current_user_id );

			if ( ! $organization->is_user_in_organization_groups( $requested_user_id ) ) {
				return new WP_REST_Response(
					array(
						'success' => false,
						'message' => __( 'Requested user is not part of the current users organization', 'ld-organization' ),
					),
					403
				);
			}

			$transient_name = 'user_certificates_' . $requested_user_id;

			$course_ids = learndash_user_get_enrolled_courses( $requested_user_id, array(), true );
			$quizzes    = get_user_meta( $requested_user_id, '_sfwd-quizzes', true );

			$data = array();

			if ( $course_ids && ! empty( $course_ids ) ) {
				foreach ( $course_ids as $course_id ) {

					$certificate_link = learndash_get_course_certificate_link( $course_id, $requested_user_id );

					if ( ! empty( $certificate_link ) ) {
						$data[] = array(
							'certificate_link_course'                            => $certificate_link,
						);
					}
				}
			}

			if ( $quizzes && ! empty( $quizzes ) ) {
				foreach ( $quizzes as $quiz_attempt ) {
					if ( isset( $quiz_attempt['certificate']['certificateLink'] ) ) {
						$data[] = array(
							'certificate_link_quiz'                            => $quiz_attempt['certificate']['certificateLink'],
						);
					}
				}
			}

			if ( ! empty( $data ) ) {
				LDOrgTransientManager::set( $transient_name, $data );
			}

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $data,
				)
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'Organization not found', 'ld-organization' ),
				),
				404
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );

	}

	/**
	 * Handles returning a list of groups for the requested users ID.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_users_groups( WP_REST_Request $request ): WP_REST_Response {
		$response = null;

		try {
			$requested_user_id = $request->get_param( 'user_id' );
			$current_user_id   = get_current_user_id();
			$organization      = Organization::by_user_id( $current_user_id );

			if ( ! $organization->is_user_in_organization_groups( $requested_user_id ) ) {
				return new WP_REST_Response(
					array(
						'success' => false,
						'message' => __( 'Requested user is not part of the current users organization', 'ld-organization' ),
					),
					403
				);
			}

			$transient_name = 'user_groups_' . $requested_user_id;

			if ( false !== ( $value = LDOrgTransientManager::get( $transient_name ) ) ) {
				return new WP_REST_Response(
					array(
						'success' => true,
						'data'    => $value,
					)
				);
			}

			$groups = learndash_get_users_group_ids( $requested_user_id );

			$data = array();
			// Loop through all groups and create a smaller data array.
			foreach ( $groups as $group_id ) {
				$group = get_post( $group_id );
				if ( ! $group ) {
					continue;
				}
				$data[] = array(
					'id'    => $group->ID,
					'title' => $group->post_title,
				);
			}

			if ( ! empty( $data ) ) {
				LDOrgTransientManager::set( $transient_name, $data );
			}

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $data,
				)
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'Organization not found', 'ld-organization' ),
				),
				404
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Fetches the requested organization and their allowed courses, if a school, otherwise it returns all courses.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 * @since 0.5.0
	 */
	final public function get_organization_allowed_courses( WP_REST_Request $request ): WP_REST_Response {
		try {
			$organization = Organization::with_id( $request->get_param( 'id' ) );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$course_ids       = $organization->get_allowed_courses_ids();
			$group_id         = $request->get_param( 'group_id' );
			$group_course_ids = learndash_get_groups_courses_ids( get_current_user_id(), array( $group_id ) );
			$final_course_ids = array_filter(
				$course_ids,
				static function ( $course_id ) use ( $group_course_ids ) {
					return ! in_array( $course_id, $group_course_ids, true );
				}
			);

			$response = new WP_REST_Response(
				array(
					'success'    => true,
					'course_ids' => array_values( $final_course_ids ),
				),
				200
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				404
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Returns the organizations groups enrollment URL.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_group_enrollment_url( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$org_id   = $request->get_param( 'id' );

		$transient_name = $org_id . '_' . $group_id . '_enrollment_url';

		if ( false !== ( $value = LDOrgTransientManager::get( $transient_name ) ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $value,
				),
				200
			);
		}
		try {
			$organization = Organization::with_id( $org_id );
			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$enrollment_url = get_signed_enrollment_url( $group_id );

			$data = array( 'enrollment_url' => $enrollment_url );

			LDOrgTransientManager::set( $transient_name, $data );

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $data,
				),
				200,
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found.',
				),
				404
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Gets all users and their progress for the requested group.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_groups_user( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );

		$transient_name = $group_id . '_group_users_progress';

		if ( false !== ( $value = LDOrgTransientManager::get( $transient_name ) ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $value,
				)
			);
		}

		$user_ids = learndash_get_groups_user_ids( $group_id, true );

		$users = array();

		foreach ( $user_ids as $user_id ) {
			$user      = get_userdata( $user_id );
			$user_data = array(
				'user_id'    => $user->ID,
				'first_name' => $user->first_name,
				'last_name'  => $user->last_name,
			);
			$progress  = learndash_get_user_group_progress( $group_id, $user_id, true );
			$status    = learndash_get_user_group_status( $group_id, $user_id );
			$users[]   = array_merge( $user_data, $progress, array( 'status' => $status ) );
		}

		if ( ! empty( $users ) ) {
			LDOrgTransientManager::set( $transient_name, $users, 5 * MINUTE_IN_SECONDS );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $users,
			)
		);
	}

	/**
	 * Handles fetching a list of group leaders for the current group.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_group_leaders( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );

		$transient_name = $group_id . 'group_group_leaders';

		if ( false !== ( $value = LDOrgTransientManager::get( $transient_name ) ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $value,
				)
			);
		}

		$group_leaders = learndash_get_groups_administrator_ids( $group_id, true );

		$users = array();

		foreach ( $group_leaders as $user_id ) {
			$user      = get_userdata( $user_id );
			$user_data = array(
				'user_id'    => $user->ID,
				'first_name' => $user->first_name,
				'last_name'  => $user->last_name,
				'email'      => $user->user_email,
			);
			$users[]   = $user_data;
		}

		if ( ! empty( $users ) ) {
			LDOrgTransientManager::set( $transient_name, $users, 5 * MINUTE_IN_SECONDS );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $users,
			)
		);
	}

	/**
	 * Handles fetching the organization based on the current user.
	 *
	 * @param WP_REST_Request $request The current request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_organization( WP_REST_Request $request ): WP_REST_Response {
		try {
			$user_id      = get_current_user_id();
			$organization = Organization::by_user_id( $user_id );
			$post         = $organization->get_organization();
			if ( null !== $post ) {
				$organization_data = array(
					'success' => true,
					'data'    => array(
						'id'   => $post->ID,
						'name' => $post->post_title,
						'slug' => $post->post_name,
						'type' => $organization->get_type(),
					),
				);
			} else {
				throw new OrganizationNotFoundException();
			}
			return new WP_REST_Response( $organization_data, 200 );
		} catch ( OrganizationNotFoundException $e ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found.',
				),
				404
			);
		}
	}

	/**
	 * Returns the number of groups in the requested organization.
	 *
	 *
	 */
	final public function get_groups( WP_REST_Request $request ): WP_REST_Response {
		try {
			$id           = $request->get_param( 'id' );
			$organization = Organization::with_id( $id );

			$groups_id     = $organization->get_organization_groups();
			$users = array();
			$groups = array();

			foreach ($groups_id as $group_id) {
				if( ! $group_id ) {
					continue;
				}

				$group = get_post($group_id);

				$users[] = learndash_get_groups_user_ids($group_id);

				$groups[] = array(
					'id' => $group->ID,
					'title' => $group->post_title,
				);

			}

			function flatten(array $array): array {
				$return = array();
				array_walk_recursive($array, function($a) use (&$return) { $return[] = $a; });
				return $return;
			}

			$users = flatten($users);
			$users = array_unique($users);
			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => array(
					'groups'  => $groups,
					'users'   => $users,
					),
					'message' => 'Successfully fetched groups for organization' . $id,
				)
			);
		} catch ( InvalidArgumentException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $exception->getMessage(),
				),
				400
			);
		} catch ( OrganizationNotFoundException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				404
			);
		}
		return rest_ensure_response( $response );
	}

	/**
	 * Returns the requested organizations licenses.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @uses \LD_Organization\Organization::with_id(),\LD_Organization\Organization::get_licenses(),\LD_Organization\Organization::get_used_licenses()
	 * @return WP_REST_Response
	 */
	final public function get_licenses( WP_REST_Request $request ): WP_REST_Response {
		try {
			$id           = $request->get_param( 'id' );
			$organization = Organization::with_id( $id );

			$used_licenses      = $organization->get_used_licenses();
			$total_licenses     = $organization->get_licenses();
			$remaining_licenses = $organization->get_remaining_licenses();

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => array(
						'used_licenses'      => $used_licenses,
						'total_licenses'     => $total_licenses,
						'remaining_licenses' => $remaining_licenses,
					),
					'message' => 'Successfully fetched licenses for organization' . $id,
				)
			);

		} catch ( InvalidArgumentException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $exception->getMessage(),
				),
				400
			);
		} catch ( OrganizationNotFoundException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				404
			);
		}
		return rest_ensure_response( $response );
	}

	/**
	 * Creates an extended information for a course based on the requesting group.
	 *
	 * @param WP_REST_Request $request The request object of the current WP_REST_Request instance.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_extended_course_information( WP_REST_Request $request ): WP_REST_Response {
		$course_id = (int) $request->get_param( 'course_id' );
		$group_id  = (int) $request->get_param( 'group_id' );

		$transient_name = 'extended_information_' . $group_id . '_' . $course_id;

		if ( false !== ( $value = LDOrgTransientManager::get( $transient_name ) ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $value,
				),
				200
			);
		}

		/** The course object @var LDLMS_Model_Course|null $course */
		$course = \LDLMS_Factory_Post::course( $course_id );
		if ( null === $course ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => __(
						'Course not found',
						'ld-organization'
					),
				),
				400
			);
		}

		$groups_ids = learndash_get_course_groups( $course_id );

		if ( ! in_array( $group_id, $groups_ids, true ) ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'This group does not have access to this course', 'ld-organization' ),
				),
				403
			);
		}
		$extended_information = array();

		$extended_information['course_progress'] = calculate_groups_course_progress( $group_id, $course_id );
		$image_url                               = get_the_post_thumbnail_url( $course_id );
		if ( '' === $image_url || false === $image_url ) {
			$image_url = get_default_image_url( 'course', 'medium' );
		}
		$extended_information['image_url'] = $image_url;

		if ( ! empty( $extended_information ) ) {
			LDOrgTransientManager::set( $transient_name, $extended_information, 5 * MINUTE_IN_SECONDS );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $extended_information,
			),
			200
		);

	}

	/**
	 * Can be used to fetch extra information about a group.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_extra_group_information( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$amount   = learndash_get_groups_user_ids( $group_id );
		$amount   = count( $amount );

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => array( 'user_amount' => $amount ),
			),
			200
		);
	}

	/**
	 * Gets the requesting users groups (depending on role param) and returns their ids.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_requesting_users_groups( WP_REST_Request $request ): WP_REST_Response {
		$type            = $request->get_param( 'role' );
		$current_user_id = get_current_user_id();
		// TODO: Maybe implement the other roles, if needed.
		if ( 'group_leader' === $type ) {
			$group_ids = learndash_get_administrators_group_ids( $current_user_id );
			$response  = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $group_ids,
				)
			);
		}

		if ( ! isset( $response ) ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'No role specified',
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Returns a list of course IDs for the requested group.
	 *
	 * @param WP_REST_Request $request The current request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_groups_course_ids( WP_REST_Request $request ): WP_REST_Response {
		$group_id    = $request->get_param( 'group_id' );
		$courses_ids = learndash_get_groups_courses_ids( get_current_user_id(), array( $group_id ) );

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $courses_ids,
			)
		);
	}

	/**
	 * Handles adding course to organization.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 * @since 0.6.0
	 */
	final public function add_course_to_organization_group( WP_REST_Request $request ): WP_REST_Response {
		$group_id      = $request->get_param( 'group_id' );
		$org_id        = $request->get_param( 'id' );
		$new_course_id = $request->get_param( 'course_id' );

		try {
			$organization = Organization::with_id( $org_id );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$allowed_courses = $organization->get_allowed_courses_ids();
			if ( ! in_array( $new_course_id, $allowed_courses, true ) ) {
				$response = new WP_REST_Response(
					array(
						'success' => false,
						'message' => 'This course is not allowed to be added to this organization',
					),
					403
				);
			} else {

				$result = ld_update_course_group_access( $new_course_id, $group_id, false );
				if ( false === $result ) {
					$response = new WP_REST_Response(
						array(
							'success' => false,
							'message' => 'Failed to add course to group',
						),
						500
					);
				} else {
					$response = new WP_REST_Response(
						array(
							'success' => true,
							'message' => 'Added course to group',
						),
						201
					);
				}
			}
		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				404
			);
		}

		return rest_ensure_response( $response );

	}

	/**
	 * Handles sending an email to a user to sign up for a course.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function invite_user_to_group( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$org_id   = $request->get_param( 'id' );
		$email    = $request->get_param( 'email' );

		$transient_name = $org_id . '_' . $group_id . '_enrollment_url';
		// phpcs:ignore WordPress.CodeAnalysis.AssignmentInCondition.Found,Squiz.PHP.DisallowMultipleAssignments.FoundInControlStructure
		if ( false !== ( $value = LDOrgTransientManager::get( $transient_name ) ) ) {
			$enrollment_data = $value;
			$group           = get_post( $group_id );
			$enrollment_data = array_merge( $enrollment_data, array( 'group_name' => $group->post_title ) );
			$success         = send_email( 'Kädenjälki <noreply@kadenjalki.fi>', $email, $enrollment_data, 'group-invite' );

			if ( $success ) {
				return new WP_REST_Response(
					array(
						'success' => true,
					),
					200
				);
			}

			return new WP_REST_Response(
				array(
					'success' => false,
				),
				500
			);
		}
		try {

			$organization = Organization::with_id( $org_id );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			if ( ! $organization->is_school() && $organization->get_remaining_licenses() <= 0 ) {
				return new WP_REST_Response(
					array(
						'success' => false,
						'message' => 'Organization is out of licenses',
					),
					403
				);
			}

			$enrollment_url = get_signed_enrollment_url( $group_id );
			$group          = get_post( $group_id );

			$data = array( 'enrollment_url' => $enrollment_url );

			LDOrgTransientManager::set( $transient_name, $data );

			$data = array_merge( $data, array( 'group_name' => $group->post_title ) );

			$success = send_email( 'Kädenjälki <noreply@kadenjalki.fi>', $email, $data, 'group-invite' );

			if ( $success ) {
				$response = new WP_REST_Response(
					array(
						'success' => true,
						'data'    => $data,
					),
					200,
				);
			} else {
				$response = new WP_REST_Response(
					array(
						'success' => false,
					),
					500
				);
			}
		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found.',
				),
				404
			);
		}

		return rest_ensure_response( $response );

	}

	/**
	 * Fetches a list of the available users for the requested groups.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 *
	 * @since 0.8.4
	 */
	final public function get_available_group_leaders( WP_REST_Request $request ): Wp_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$org_id   = $request->get_param( 'id' );

		try {
			$organization = Organization::with_id( $org_id );
			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$group_leaders = $organization->get_group_leaders();

			$admin_ids = learndash_get_groups_administrator_ids( $group_id, true );

			$not_in_group = array_filter(
				$group_leaders,
				static function ( $leader ) use ( $admin_ids ) {
					return ( ! in_array( $leader->ID, $admin_ids, true ) );
				}
			);

			$available_leaders = array();

			foreach ( $not_in_group as $leader ) {
				$available_leaders[] = array(
					'id'    => $leader->ID,
					'name'  => $leader->first_name . ' ' . $leader->last_name,
					'email' => $leader->user_email,
				);
			}

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'message' => __( 'Fetched group leaders successfully', 'ld-organization' ),
					'data'    => $available_leaders,
				),
				200
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				404
			);
		}

		return rest_ensure_response( $response );

	}

	/**
	 * Adds a valid user to the group.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function add_leader_to_group( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$org_id   = $request->get_param( 'id' );
		$user_id  = $request->get_param( 'user_id' );

		try {
			$organization = Organization::with_id( $org_id );
			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			if ( ! $organization->has_user( $user_id, 'group_leaders', true ) ) {
				throw new InvalidArgumentException( __( 'User is not in the organization', 'ld-organization' ) );
			}

			if ( ! user_has_role( $user_id, array( 'group_leader', 'opettaja', 'administrator' ) ) ) {
				throw new InvalidArgumentException( __( 'Selected user does not have the correct permissions', 'ld-organization' ) );
			}

			$admin_ids = learndash_get_groups_administrator_ids( $group_id, true );

			if ( in_array( $user_id, $admin_ids, true ) ) {
				throw new InvalidArgumentException( __( 'User already apart of the group.', 'ld-organization' ) );
			}

			$success = ld_update_leader_group_access( $user_id, $group_id );

			if ( $success ) {
				$response = new WP_REST_Response(
					array(
						'success' => true,
						'message' => __( 'User successfully added', 'ld-organization' ),
					),
					201
				);

				LDOrgTransientManager::delete( $group_id . 'group_group_leaders' );
			} else {
				$response = new WP_REST_Response(
					array(
						'success' => false,
						'message' => __( 'Unable to add user as group leader to this group', 'ld-organization' ),
					),
					500
				);
			}
		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				404
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}
		return rest_ensure_response( $response );
	}

	/**
	 * Either creates or adds a user to the organization.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function invite_group_leader( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$org_id   = $request->get_param( 'id' );

		$user_data = array(
			'email'      => $request->get_param( 'user_email' ),
			'user_email' => $request->get_param( 'user_email' ),
			'first_name' => $request->get_param( 'first_name' ),
			'last_name'  => $request->get_param( 'last_name' ),
		);

		try {
			$organization = Organization::with_id( $org_id );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			if ( ! $organization->is_school() && $organization->get_remaining_licenses() <= 0 ) {
				return new WP_REST_Response(
					array(
						'success' => false,
						'message' => 'Organization is out of licenses',
					),
					403
				);
			}

			$user = email_exists( $user_data['email'] );

			if ( ! $user ) {
				$user_data['user_pass']  = wp_generate_password();
				$user_data['user_login'] = $user_data['user_email'];
				$new_user_id             = wp_insert_user( $user_data );
				if ( is_wp_error( $new_user_id ) ) {
					return new WP_REST_Response(
						array(
							'success' => false,
							'message' => $new_user_id->get_error_message(),
						),
						500
					);
				}

				wp_new_user_notification( $new_user_id );
				$user = get_user_by( 'ID', $new_user_id );

			} else {
				$has_organizations = get_users_organization_id( $user );

				if ( ! empty( $has_organizations ) && $has_organizations[0] !== $organization->get_organization()->ID ) {
					return new WP_REST_Response(
						array(
							'success' => false,
							'message' => 'User already belongs to an organization.',
						),
						400
					);
				}

				$user = get_user_by( 'email', $user_data['email'] );
			}

			$organization->add_user_to_organization( $user->ID, 'group_leaders' );

			if ( $organization->is_school() ) {
				$user->add_role( 'opettaja' );
			}
			$user->add_role( 'group_leader' );

			ld_update_leader_group_access( $user->ID, $group_id );

			$group = get_post( $group_id );

			$email_data = array(
				'group_name' => $group->post_title,
				'user_name'  => $user->first_name,
			);

			send_email( 'Kädenjälki <noreply@kadenjalki.fi>', $user->user_email, $email_data, 'new-group-leader' );
			$response = new WP_REST_Response(
				array(
					'success' => true,
					'message' => 'Added new group leader.',
				),
				200,
			);

			LDOrgTransientManager::delete( $group_id . 'group_group_leaders' );

		} catch ( PermissionException | OrganizationNotFoundException | InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				404
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * PERMISSION CALLBACKS
	 */

	/**
	 * Checks if the user is logged in.
	 *
	 * @return bool
	 */
	final public function is_logged_in(): bool {
		return is_user_logged_in();
	}

	/**
	 * Handles checking if the user is logged in and group leader for the requested course.
	 *
	 * @param WP_REST_Request $request The current WP_REST_Request object.
	 *
	 * @return bool
	 */
	final public function is_group_leader_of_post( WP_REST_Request $request ): bool {
		if ( ! is_user_logged_in() ) {
			return false;
		}

		$current_user = wp_get_current_user();

		if ( null === $current_user ) {
			return false;
		}

		// Current user is an admin user, allow all.
		if ( current_user_can( 'manage_options' ) ) {
			return true;
		}

		// If we pass the WP checks, we check if the user is a group leader and is a part of the group administrators.
		$admins = learndash_get_groups_administrator_ids( $request->get_param( 'group_id' ) );

		if ( ! in_array( $current_user->ID, $admins, true ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Checks if the user calling the API is an owner of the organization being called.
	 *
	 * @param WP_REST_Request $request The request object for the current request.
	 * @param string          $param_key The parameter key to check.
	 *
	 * @return bool
	 * @uses \LD_Organization\RestAPI::has_access()
	 */
	final public function is_owner( WP_REST_Request $request, string $param_key = 'id' ): bool {
		return $this->has_access( $request, 'organization_owners', $param_key );
	}

	/**
	 * Checks if the user calling the API is an owner of the organization being called.
	 *
	 * @param WP_REST_Request $request The request object for the current request.
	 * @param string          $param_key The parameter key to check for organization.
	 *
	 * @return bool
	 * @uses \LD_Organization\RestAPI::has_access()
	 */
	final public function is_group_leader( WP_REST_Request $request, string $param_key = 'id' ): bool {
		return $this->has_access( $request, 'group_leaders', $param_key );
	}

	/**
	 * Handles checking if the organization is owned by the authenticated user.
	 *
	 * @param WP_REST_Request $request The request with a required parameter of ID.
	 * @param string          $role The role to check against.
	 * @param string          $param_key Optionally the key of the parameter to check.
	 *
	 * @return bool
	 * @uses \LD_Organization\Organization::with_id(),\LD_Organization\Organization::has_user()
	 */
	final public function has_access( WP_REST_Request $request, string $role = 'both', string $param_key = 'id' ): bool {
		if ( ! is_user_logged_in() ) {
			return false;
		}
		$id = $request->get_param( $param_key );
		if ( ! isset( $id ) ) {
			return false;
		}
		try {
			$organization = Organization::with_id( $id );
			$current_user = wp_get_current_user();
			if ( isset( $current_user ) && $organization->has_user( $current_user->ID, $role ) ) {
				return true;
			}
		} catch ( OrganizationNotFoundException | InvalidArgumentException $exception ) {
			return false;
		}
		return false;
	}

	/**
	 *  VALIDATION CALLBACKS.
	 */

	/**
	 * The default validate_callback if the endpoint has an ID in it.
	 *
	 * @return array
	 */
	final public function validate_id_argument(): array {
		return array(
			'id' => array(
				'validate_callback' => function( $param, $request, $key ) {
					return is_numeric( $param );
				},
			),
		);
	}

	/**
	 * Checks if the parameter is a valid group_id.
	 *
	 * @return array
	 */
	final public function is_valid_ld_group_id(): array {
		return array(
			'validate_callback' => function( $param, $request, $key ) {
				return is_numeric( $param ) && learndash_is_group_post( $param );
			},
		);
	}

	/**
	 * Checks if the parameter is a valid User ID.
	 *
	 * @return array
	 */
	final public function is_valid_user_id(): array {
		return array(
			'validate_callback' => function( $param, $request, $key ) {
				return is_numeric( $param ) && false !== get_user_by( 'ID', $param );
			},
		);
	}

	/**
	 * Checks if the parameter is a valid group post ID.
	 *
	 * @return array
	 */
	final public function is_valid_ld_course_id(): array {
		return array(
			'validate_callback' => function( $param, $request, $key ) {
				return is_numeric( $param ) && learndash_is_course_post( $param );
			},
		);
	}

	/**
	 * Checks if the parameter is a valid organization ID.
	 *
	 * @return array
	 */
	final public function is_valid_org_id(): array {
		return array(
			'validate_callback' => function( $param, $request, $key ) {
				return is_numeric( $param ) && is_valid_organization_post( $param );
			},
		);
	}

	/**
	 * GETTERS
	 */

	/**
	 * Gets the namespace string for the Rest API.
	 *
	 * @return string
	 */
	final public function get_namespace(): string {
		return $this->namespace . '/' . $this->version;
	}

	/**
	 * Returns the array of valid roles that can be requested.
	 *
	 * @return string[]
	 */
	private function get_valid_roles_array(): array {
		return array( 'group_leader', 'organization_owner' );
	}

	/**
	 * Registers the routes for handling organization stuff.
	 *
	 * @return void
	 */
	final public function register_organization_routes(): void {
		register_rest_route(
			$this->get_namespace(),
			'org/',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_organization' ),
				'permission_callback' => array( $this, 'is_logged_in' ),
			)
		);
		register_rest_route($this->get_namespace(),
			'org/(?P<id>\d+)/courses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_organization_courses' ),
				'permission_callback' => array( $this, 'is_owner' ),
				'args'                => $this->validate_id_argument(),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/new_courses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_organization_allowed_courses' ),
				'permission_callback' => array( $this, 'is_group_leader' ),
				'args'                => array(
					'id'       => $this->is_valid_org_id(),
					'group_id' => $this->is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/courses',
			array(
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => array( $this, 'add_course_to_organization_group' ),
				'permission_callback' => array( $this, 'is_group_leader' ),
				'args'                => array(
					'id'        => $this->is_valid_org_id(),
					'group_id'  => $this->is_valid_ld_group_id(),
					'course_id' => $this->is_valid_ld_course_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/enrollment',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_group_enrollment_url' ),
				'permission_callback' => array( $this, 'is_group_leader' ),
				'args'                => array(
					'id'       => $this->is_valid_org_id(),
					'group_id' => $this->is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/licenses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_licenses' ),
				'permission_callback' => array( $this, 'is_owner' ),
				'args'                => $this->validate_id_argument(),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/invite',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'invite_user_to_group' ),
				'permission_callback' => array( $this, 'is_group_leader' ),
				'args'                => array(
					'group_id' => $this->is_valid_ld_group_id(),
					'id'       => $this->is_valid_org_id(),
					'email'    => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param ) && false !== filter_var( trim( $param ), FILTER_VALIDATE_EMAIL );
						},
						'sanitize_callback' => function ( $param, $request, $key ) {
							return sanitize_email( $param );
						},
					),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/available_leaders',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_available_group_leaders' ),
				'permission_callback' => array( $this, 'is_group_leader' ),
				'args'                => array(
					'id'       => $this->is_valid_org_id(),
					'group_id' => $this->is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/add_leader',
			array(
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => array( $this, 'add_leader_to_group' ),
				'permission_callback' => array( $this, 'is_group_leader' ),
				'args'                => array(
					'group_id' => $this->is_valid_ld_group_id(),
					'id'       => $this->is_valid_org_id(),
					'user_id'  => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_numeric( $param ) && false !== get_user_by( 'ID', $param );
						},
					),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/add_leader_deprecated',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'invite_group_leader' ),
				'permission_callback' => '__return_false',
				'args'                => array(
					'group_id'   => $this->is_valid_ld_group_id(),
					'id'         => $this->is_valid_org_id(),
					'user_email' => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param ) && false !== filter_var( trim( $param ), FILTER_VALIDATE_EMAIL );
						},
						'sanitize_callback' => function ( $param, $request, $key ) {
							return sanitize_email( $param );
						},
					),
					'first_name' => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param );
						},
						'sanitize_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
					'last_name'  => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param );
						},
						'sanitize_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
				),
			)
		);
	}

	/**
	 * Registers the routes for groups.
	 *
	 * @return void
	 */
	final public function register_groups_routes(): void {
		register_rest_route(
			$this->get_namespace(),
			'groups',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_requesting_users_groups' ),
				'permission_callback' => array( $this, 'is_logged_in' ),
				'args'                => array(
					'role' => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param ) && in_array( (string) $param, $this->get_valid_roles_array(), true );
						},
					),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/course/(?P<course_id>\d+)/information',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_extended_course_information' ),
				'permission_callback' => array( $this, 'is_group_leader_of_post' ),
				'args'                => array(
					'course_id' => $this->is_valid_ld_course_id(),
					'group_id'  => $this->is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/extra',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_extra_group_information' ),
				'permission_callback' => array( $this, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id' => $this->is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/courses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_groups_course_ids' ),
				'permission_callback' => array( $this, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id' => $this->is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/users',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_groups_user' ),
				'permission_callback' => array( $this, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id' => $this->is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/group_leaders',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_group_leaders' ),
				'permission_callback' => array( $this, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id' => $this->is_valid_ld_group_id(),
				),
			)
		);
	}

	/**
	 * Registers the user routes.
	 *
	 * @return void
	 */
	final public function register_user_routes(): void {
		register_rest_route(
			$this->get_namespace(),
			'user/(?P<user_id>\d+)/groups',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_users_groups' ),
				'permission_callback' => array( $this, 'is_logged_in' ),
				'args'                => array(
					'user_id' => $this->is_valid_user_id(),
				),
			)
		);

		register_rest_route(
			$this->get_namespace(),
			'user/(?P<user_id>\d+)/groups/(?P<group_id>\d+)/course_progress',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_user_group_progress' ),
				'permission_callback' => array( $this, 'is_logged_in' ),
				'args'                => array(
					'user_id'  => $this->is_valid_user_id(),
					'group_id' => $this->is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'user/(?P<user_id>\d+)/certificates',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_user_certificates' ),
				'permission_callback' => array( $this, 'is_logged_in' ),
				'args'                => array(
					'user_id'  => $this->is_valid_user_id(),
				),
			)
		);
	}
}
