<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
// phpcs:disable Generic.Commenting.DocComment.MissingShort
/**
 * Handles adding fields to the REST fields.
 *
 * @package LD_Organization\Hooks
 */

namespace LD_Organization\Hooks;

use LD_Organization\LDOrgTransientManager;
use WP_REST_Request;
use function LD_Organization\calculate_groups_course_progress;
use function LD_Organization\get_default_image_url;

/**
 * Adds fields to the Course rest API object that are needed in the frontend.
 *
 * @package LD_Organization\Hooks
 */
class CourseRestFields extends AbstractHook {
	/**
	 * Handles adding the fields to the rest API object for sfwd-course.
	 *
	 * @return void
	 * @uses learndash_get_post_type_slug
	 */
	final public function add_rest_fields(): void {
		$course_post_type = learndash_get_post_type_slug( 'course', 'string' );
		register_rest_field( $course_post_type, 'course_progress', array( 'get_callback' => array( $this, 'get_course_progress' ) ) );
		register_rest_field( $course_post_type, 'image_url', array( 'get_callback' => array( $this, 'get_course_thumbnail' ) ) );
	}

	/**
	 * Handles fetching additional fields for Course (group progress) Rest API object.
	 *
	 * @param array           $course The prepared course from the Rest API object.
	 * @param string          $attr The attribute name.
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return array
	 */
	final public function get_course_progress( array $course, string $attr, WP_REST_Request $request ): array {
		$group_id = $request->get_param( 'group_id' );
		if ( ! empty( $course['id'] ) && isset( $group_id ) ) {
			$group_id  = absint( $group_id );
			$course_id = $course['id'];

			if ( ! learndash_is_group_post( $group_id ) ) {
				return array();
			}

			$transient_name = 'extended_information_' . $group_id . '_' . $course_id;

			if ( false !== ( $value = LDOrgTransientManager::get( $transient_name ) ) ) {
				return $value;
			}

			$groups_ids = learndash_get_course_groups( $course_id );

			if ( ! in_array( $group_id, $groups_ids, true ) ) {
				return array();
			}

			$information = calculate_groups_course_progress( $group_id, $course_id );

			if ( ! empty( $information ) ) {
				LDOrgTransientManager::set( $transient_name, $information, 5 * MINUTE_IN_SECONDS );
			}

			return $information;

		}
		return array();
	}

	/**
	 * Handles fetching the thumbnail for a given course.
	 *
	 * @param array $course The prepared course object.
	 *
	 * @return string
	 */
	final public function get_course_thumbnail( array $course ): string {
		if ( ! empty( $course['id'] ) ) {
			$image_url = get_the_post_thumbnail_url( $course['id'] );
			if ( '' === $image_url || false === $image_url ) {
				$image_url = get_default_image_url( 'course', 'medium' );
			}
			return $image_url;
		}
		return '';
	}

	/**
	 * @inheritDoc
	 */
	final public function get_actions(): array {
		return array(
			array(
				'hook'     => 'rest_api_init',
				'callable' => array( $this, 'add_rest_fields' ),
				'priority' => 99,
				'num_args' => 0,
			),
		);
	}

	/**
	 * @inheritDoc
	 */
	final public function get_filters(): array {
		return array();
	}
}
