<?php
/**
 * Handles registering the RestAPI endpoints for handling organizations.
 *
 * @package LD_Organization
 */

namespace LD_Organization;

use InvalidArgumentException;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * RestAPI class, when initialized will register the restAPI.
 *
 * @package LD_Organization
 * @since 0.2.0
 */
class RestAPI {

	/**
	 * The RestAPI namespace.
	 *
	 * @var string
	 */
	private string $namespace = 'ld-organization';

	/**
	 * The RestAPI version.
	 *
	 * @var string
	 */
	private string $version = 'v1';

	/**
	 * Registers all required functionality and endpoints.
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_endpoints' ) );
	}

	/**
	 * The default validate_callback if the endpoint has an ID in it.
	 *
	 * @return array
	 */
	final public function validate_id_argument(): array {
		return array(
			'id' => array(
				'validate_callback' => function( $param, $request, $key ) {
					return is_numeric( $param );
				},
			),
		);
	}

	/**
	 * Registers all the endpoint and their parameters.
	 *
	 * @return void
	 */
	final public function register_endpoints(): void {
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/licenses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_licenses' ),
				'permission_callback' => array( $this, 'is_owner' ),
				'args'                => $this->validate_id_argument(),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_organization' ),
				'permission_callback' => array( $this, 'is_logged_in' ),
			)
		);
	}

	/**
	 * Checks if the user is logged in.
	 *
	 * @return bool
	 */
	final public function is_logged_in(): bool {
		return is_user_logged_in();
	}

	/**
	 * Checks if the user calling the API is an owner of the organization being called.
	 *
	 * @param WP_REST_Request $request The request object for the current request.
	 *
	 * @uses \LD_Organization\RestAPI::has_access()
	 * @return bool
	 */
	final public function is_owner( WP_REST_Request $request ): bool {
		return $this->has_access( $request, 'organization_owners' );
	}

	/**
	 * Checks if the user calling the API is an owner of the organization being called.
	 *
	 * @param WP_REST_Request $request The request object for the current request.
	 *
	 * @uses \LD_Organization\RestAPI::has_access()
	 * @return bool
	 */
	final public function is_group_leader( WP_REST_Request $request ): bool {
		return $this->has_access( $request, 'group_leaders' );
	}

	/**
	 * Handles checking if the organization is owned by the authenticated user.
	 *
	 * @param WP_REST_Request $request The request with a required parameter of ID.
	 * @param string          $role The role to check against.
	 *
	 * @uses \LD_Organization\Organization::with_id(),\LD_Organization\Organization::has_user()
	 * @return bool
	 */
	final public function has_access( WP_REST_Request $request, string $role = 'both' ): bool {
		if ( ! is_user_logged_in() ) {
			return false;
		}
		$id = $request->get_param( 'id' );
		if ( ! isset( $id ) ) {
			return false;
		}
		try {
			$organization = Organization::with_id( $id );
			$current_user = wp_get_current_user();
			if ( isset( $current_user ) && $organization->has_user( $current_user->ID, $role ) ) {
				return true;
			}
		} catch ( OrganizationNotFoundException | InvalidArgumentException $exception ) {
			return false;
		}
		return false;
	}

	/**
	 * Handles fetching the organization based on the current user.
	 *
	 * @param WP_REST_Request $request The current request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_organization( WP_REST_Request $request ): WP_REST_Response {
		try {
			$user_id           = wp_get_current_user()->ID;
			$organization      = Organization::by_user_id( $user_id );
			$organization      = $organization->get_organization();
			$organization_data = array(
				'success' => true,
				'data'    => array(
					'id'   => $organization->ID,
					'name' => $organization->post_title,
					'slug' => $organization->post_name,
				),
			);
			return new WP_REST_Response( $organization_data, 200 );
		} catch ( OrganizationNotFoundException $e ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'User does not belong to any organization',
				),
				403
			);
		}
	}

	/**
	 * Returns the requested organizations licenses.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @uses \LD_Organization\Organization::with_id(),\LD_Organization\Organization::get_licenses(),\LD_Organization\Organization::get_used_licenses()
	 * @return WP_REST_Response
	 */
	final public function get_licenses( WP_REST_Request $request ): WP_REST_Response {
		try {
			$id           = $request->get_param( 'id' );
			$organization = Organization::with_id( $id );

			$used_licenses  = $organization->get_used_licenses();
			$total_licenses = $organization->get_licenses();

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => array(
						'used_licenses'  => $used_licenses,
						'total_licenses' => $total_licenses,
					),
					'message' => 'Successfully fetched licenses for organization' . $id,
				)
			);

		} catch ( InvalidArgumentException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $exception->getMessage(),
				),
				400
			);
		} catch ( OrganizationNotFoundException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				400
			);
		}
		return rest_ensure_response( $response );
	}

	/**
	 * Gets the namespace string for the Rest API.
	 *
	 * @return string
	 */
	final public function get_namespace(): string {
		return $this->namespace . '/' . $this->version;
	}
}
