<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
namespace LD_Organization\Data;

use DateTime;
use DateTimeZone;
use Exception;
use LD_Organization\Hooks\ExportEssayAjax;
use const LD_Organization\LD_ORG_BASE_PATH;

/**
 * Class EssayExportProcessingData
 *
 * The data object for the export_essay ajax request.
 *
 * @package LD_Organization\Data
 * @since 0.20.0
 */
class EssayExportProcessingData {

	/**
	 * Processing prefix.
	 *
	 * @var string
	 */
	private const PROCESSING_PREFIX = '_processing_data_';

	/**
	 * The timeout for the processing data transient.
	 *
	 * @var int
	 */
	private const PROCESSING_DATA_TIMEOUT = 5 * DAY_IN_SECONDS;

	/**
	 * The processing id for the export_essay ajax request.
	 *
	 * @var string
	 */
	public string $processing_id;

	/**
	 * The requesting user id.
	 *
	 * @var int
	 */
	public int $user_id;

	/**
	 * Current iteration.
	 *
	 * @var int
	 */
	public int $iteration = 0;

	/**
	 * Is the export done?
	 *
	 * @var bool
	 */
	public bool $done = false;

	/**
	 * Group IDs.
	 *
	 * @var int[]
	 */
	public array $group_ids = array();

	/**
	 * Has manually selected group IDs.
	 *
	 * @var bool
	 */
	public bool $manual_groups = false;

	/**
	 * User IDs.
	 *
	 * @var int[]
	 */
	public array $user_ids = array();

	/**
	 * Course IDs.
	 *
	 * @var int[]
	 */
	public array $course_ids = array();

	/**
	 * The start date.
	 *
	 * @var ?DateTime
	 */
	public ?DateTime $start_date;

	/**
	 * The end date.
	 *
	 * @var ?DateTime
	 */
	public ?DateTime $end_date;
	/**
	 * Is the export empty?
	 *
	 * @var bool
	 */
	public bool $empty = false;

	/**
	 * Total posts.
	 *
	 * @var int The total number of posts.
	 */
	public int $total_posts = 0;

	/**
	 * Per page.
	 *
	 * @var int The number of posts per page.
	 */
	public int $per_page = ExportEssayAjax::PER_PAGE;

	/**
	 * The file path.
	 *
	 * @var string
	 */
	private string $file_path;
	/**
	 * The file name.
	 *
	 * @var string
	 */
	private string $file_name;
	/**
	 * The last saved time.
	 *
	 * @var int|null
	 */
	private ?int $last_saved;

	/**
	 * EssayExportProcessingData constructor.
	 *
	 * @param string $processing_id The processing id.
	 * @param int $user_id The requesting user id.
	 * @param ?DateTime $start_date The start date.
	 * @param ?DateTime $end_date The end date.
	 * @param ?string $file_path The file path.
	 * @param ?string $file_name The file name.
	 */
	public function __construct( string $processing_id, int $user_id, ?DateTime $start_date = null, ?DateTime $end_date = null, ?string $file_path = null, ?string $file_name = null ) {
		$this->processing_id = $processing_id;
		$this->user_id       = $user_id;
		$this->start_date    = $start_date;
		$this->end_date      = $end_date;
		$this->file_path     = trailingslashit( LD_ORG_BASE_PATH ) . 'export/';
		if ( empty( $file_name ) ) {
			$file_name = 'export-' . $this->processing_id . '.csv';
		}
		$this->file_name = $file_name;
	}

	/**
	 * Checks if the export is done.
	 *
	 * @return bool
	 */
	final public function is_done(): bool {
		return $this->done;
	}

	/**
	 * Sets the export done.
	 *
	 * @param bool $is_empty Is the export empty.
	 *
	 * @return void
	 */
	final public function set_done( bool $is_empty = false ): void {
		if ( $is_empty ) {
			$this->empty = true;
		}
		$this->done = true;
	}

	/**
	 * Handles getting the iteration.
	 *
	 * @return int
	 */
	final public function get_iteration(): int {
		return $this->iteration;
	}

	/**
	 * Increments the iteration.
	 *
	 * @return void
	 */
	final public function increment_iteration(): void {
		++ $this->iteration;
	}

	/**
	 * Handles saving the processing data.
	 *
	 * @return bool
	 */
	final public function save(): bool {
		$transient_key    = self::PROCESSING_PREFIX . $this->processing_id;
		$this->last_saved = time();

		return set_transient( $transient_key, $this, self::PROCESSING_DATA_TIMEOUT );
	}

	/**
	 * Handles getting the last saved date.
	 *
	 * @param string $format The date format.
	 *
	 * @return string
	 */
	final public function get_last_saved( string $format = 'd.m.Y' ): string {
		$last_saved = $this->last_saved;
		if ( is_numeric( $last_saved ) ) {
			try {
				return ( new DateTime( '@' . $last_saved, new DateTimeZone( 'Europe/Helsinki' ) ) )->format( $format );
			} catch ( Exception $e ) {
				logger( 'EssayExportProcessingData' )->error( 'Failed to get last saved date.', array( 'error' => $e->getMessage() ) );
			}
		}
		try {
			return ( new DateTime( 'now', new DateTimeZone( 'Europe/Helsinki' ) ) )->format( $format );
		} catch ( Exception $e ) {
			logger( 'EssayExportProcessingData' )->error( 'Failed to get last saved date.', array( 'error' => $e->getMessage() ) );
		}

		return '';
	}

	/**
	 * Handles getting the file path.
	 *
	 * @return bool|string
	 */
	final public function get_file(): bool|string {
		$file_path    = trailingslashit( $this->file_path );
		$allowed_path = trailingslashit( LD_ORG_BASE_PATH ) . 'export/';
		if ( $file_path !== $allowed_path ) {
			logger( 'EssayExportProcessingData' )->error(
				'File path is not allowed.',
				array(
					'file_path'    => $file_path,
					'allowed_path' => $allowed_path,
				)
			);

			return false;
		}
		if ( ! is_dir( $file_path ) ) {
			$success = wp_mkdir_p( $file_path );
			if ( ! $success ) {
				logger( 'EssayExportProcessingData' )->error(
					'Failed to create export directory.',
					array(
						'file_path' => $file_path,
					)
				);

				return false;
			}
		}
		$file_path = trailingslashit( $file_path ) . $this->file_name;
		if ( validate_file( $file_path ) > 0 ) {
			logger( 'EssayExportProcessingData' )->error(
				'File path is not allowed.',
				array(
					'file_path'     => $file_path,
					'validate_file' => validate_file( $file_path ),
				)
			);

			return false;
		}
		if ( ! is_file( $file_path ) ) {
			// phpcs:disable WordPress.WP.AlternativeFunctions
			$file = fopen( $file_path, 'wb' );
			if ( ! $file ) {
				logger( 'EssayExportProcessingData' )->error(
					'Failed to open file.',
					array(
						'file_path' => $file_path,
					)
				);

				return false;
			}
			fclose( $file );
			// phpcs:enable WordPress.WP.AlternativeFunctions
		}
		if ( realpath( $file_path ) === false || realpath( $file_path ) !== $file_path ) {
			logger( 'EssayExportProcessingData' )->error(
				'File path is not allowed.',
				array(
					'file_path'    => $file_path,
					'allowed_path' => $allowed_path,
				)
			);

			return false;
		}

		return $file_path;
	}
}
