<?php
/**
 * Certain helper functions for use in the plugin.
 *
 * @package LD_Organization
 * @since 0.4.0
 */

namespace LD_Organization;

use WP_Post;

/**
 * Handles fetching the default set image for a given post type.
 *
 * @param string $post_type The post type to fetch the default image for.
 * @param string $size The size (WordPress size) to use.
 *
 * @return string
 */
function get_default_image_url( string $post_type = '', string $size = 'full' ): string {
	$default_image_id = get_field( 'ld_organization_default_image_url_' . $post_type, 'options' );

	if ( isset( $default_image_id ) ) {
		$image_src_array = wp_get_attachment_image_src( $default_image_id, $size );
		if ( ( ! empty( $image_src_array ) ) && ( is_array( $image_src_array ) ) && ( ! empty( $image_src_array[0] ) ) ) {
			return $image_src_array[0];
		}
	}

	return '';
}


/**
 * Checks if the given post is an organization post.
 *
 * @param WP_Post|int $post The post object or post ID.
 *
 * @return bool
 */
function is_valid_organization_post( $post ): bool {
	if ( empty( $post ) ) {
		return false;
	}

	$post_type = is_a( $post, WP_Post::class ) ? $post->post_type : get_post_type( $post );

	return POST_TYPE_NAME === $post_type;
}

/**
 * Converts INF/NaN to 0.
 *
 * @param mixed $value The value to check against.
 *
 * @return mixed
 */
function handle_erroneous_values( $value ) {
	return ( is_nan( $value ) || is_infinite( $value ) ) ? 0 : $value;
}



/**
 * Handles the sending of emails using the Notification class
 *
 * @param string $from_email From email.
 * @param string $to_email To email.
 * @param array  $data The data to be added if applicable.
 * @param string $type The slug of CPT notification that should be sent.
 * @param bool   $use_recipients Should the receivers be fetched from the template, overrides $to_email.
 *
 * @return bool Indicates if the sending was successful.
 * @uses \LD_Organization\Notification
 */
function send_email( string $from_email, string $to_email, array $data = array(), string $type = 'default', bool $use_recipients = false ): bool {

	// Otherwise construct the notification and send it.
	try {
		$notification = new Notification( $type );
		$notification->set_from( $from_email );
		if ( ! empty( $data ) ) {
			$notification->set_data( $data );
		}
		if ( ! $use_recipients ) {
			$notification->set_to( $to_email );
			return $notification->send();
		}

		return $notification->send( $notification->get_recipients() );

	} catch ( \Throwable $th ) {
		return false;
	}
}

/**
 * Fetches the users organization if it exists.
 *
 * @param int $user_id The user ID.
 *
 * @return int[]
 */
function get_users_organization_id( int $user_id ) {
	return get_posts(
		array(
			'post_type'   => Organization::$post_type,
			'numberposts' => 1,
			'fields'      => 'ids',
			'meta_query'  => array( // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
				'relation' => 'OR',
				array(
					'key'     => 'organization_owners',
					'value'   => '"' . $user_id . '"',
					'compare' => 'LIKE',
				),
				array(
					'key'     => 'group_leaders',
					'value'   => '"' . $user_id . '"',
					'compare' => 'LIKE',
				),
			),
		)
	);
}
