<?php
/**
 * Handles registering different hooks that hook into the creation/updating of organization post types
 * For example setting the used licenses to 0 on creation.
 *
 * @package LD_Organization
 */

namespace LD_Organization;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use WP_Post;

/**
 * Handles adding 0 as the licenses used for newly created organization CPTs.
 *
 * @param int     $post_id The post ID for the inserted post.
 * @param WP_Post $post The post object.
 * @param bool    $update If this is an update or not.
 *
 * @return void
 */
function add_licenses_on_create( int $post_id, WP_Post $post, bool $update ) {
	// Bail early if we are updating.
	if ( $update ) {
		return;
	}
	// Check if it is the correct post type and the value is not somehow set.
	$used_licenses = get_post_meta( $post_id, USED_LICENSES_META_KEY, true );
	if ( POST_TYPE_NAME === $post->post_type && ( 0 !== $used_licenses || empty( $used_licenses ) ) ) {
		update_post_meta( $post_id, USED_LICENSES_META_KEY, 0, 0 );
	}
}

add_action( 'wp_insert_post', 'LD_Organization\add_licenses_on_create', 10, 3 );


/**
 * Handles the clearing of transients on post update.
 *
 * @param int     $post_id The post ID for the inserted post.
 * @param WP_Post $post The post object.
 * @param bool    $update If this is an update or not.
 *
 * @return void
 * @since 0.5.0
 */
function clear_organization_transients( int $post_id, WP_Post $post, bool $update ) {
	foreach ( Organization::$transients as $key => $transient_key ) {
		$name = Organization::get_transient_name( $transient_key, $post_id );
		if ( ! empty( $name ) ) {
			TransientManager::delete( $name );
		}
	}
}

add_action( 'save_post_organization', 'LD_Organization\clear_organization_transients', 10, 3 );
