<?php

namespace LD_Organization;

use http\Exception\InvalidArgumentException;
use WP_Post;

/**
 * The Organization class is responsible fetching and checking different attributes revolving around organizations,
 * can and should be used when directly communicating with Learndash.
 *
 * @package LD_Organization
 */
class Organization {

	/**
	 * The post type slug to use.
	 * Default: "organization".
	 *
	 * @var string
	 */
	private string $post_type = POST_TYPE_NAME;

	/**
	 * The organization being handled.
	 *
	 * @var WP_Post|null
	 */
	private WP_Post $organization;

	/**
	 * TODO: Needs a constructor?
	 */
	public function __construct() {
	}

	/**
	 * Handles initializing Organization class with ID.
	 *
	 * @param int $id The ID of the organization to retrieve.
	 *
	 * @return Organization
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws \InvalidArgumentException If the ID belongs to a non-organization post.
	 */
	public static function with_id( int $id ): Organization {
		$instance = new self();
		$post     = $instance->get_post_by_id( $id );
		$instance->set_organization( $post );
		return $instance;
	}

	/**
	 * Handles initializing Organization class based on a user ID.
	 *
	 * @param int    $id The ID of a user belonging to the organization to retrieve.
	 * @param string $key The meta key to query on, uses Organization::get_post_by_user_id so the accepted values are the same as that.
	 *
	 * @return Organization
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws \InvalidArgumentException If the ID belongs to a non-organization post or the passed key is invalid.
	 *
	 * @uses Organization::get_post_by_user_id()
	 */
	public static function by_user_id( int $id, string $key = 'both' ): Organization {
		$instance = new self();
		$post     = $instance->get_post_by_user_id( $id, $key );
		$instance->set_organization( $post );
		return $instance;
	}

	/**
	 * Handles initializing Organization class based on a user ID.
	 *
	 * @param string $email The email of the user belonging to an organization.
	 * @param string $key The meta key to query on, uses Organization::get_post_by_user_id so the accepted values are the same as that.
	 *
	 * @return Organization
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws \InvalidArgumentException If the ID belongs to a non-organization post or the passed key is invalid.
	 *
	 * @uses Organization::get_post_by_user_email()
	 */
	public static function by_user_email( string $email, string $key = 'both' ): Organization {
		$instance = new self();
		$post     = $instance->get_post_by_user_email( $email, $key );
		$instance->set_organization( $post );
		return $instance;
	}

	/**
	 * Handles initializing Organization class based on the user ID.
	 *
	 * @param string $email The email to lookup user by.
	 * @param string $key The meta key to query on, uses Organization::get_post_by_user_id so the accepted values are the same as that.
	 *
	 * @return WP_Post
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws \InvalidArgumentException If a user is not found, because the email is not found.
	 *
	 * @uses Organization::get_post_by_user_id()
	 */
	private function get_post_by_user_email( string $email, string $key = 'both' ): WP_Post {
		$user = get_user_by( 'email', $email );
		if ( ! $user ) {
			throw new \InvalidArgumentException( 'No user with the provided email found' );
		}
		return $this->get_post_by_user_id( $user->ID, $key );
	}

	/**
	 * Handles fetching the post by ID.
	 *
	 * @param int $id The Post id.
	 *
	 * @return WP_Post
	 * @throws OrganizationNotFoundException Throws this exception if a Organization cannot be found.
	 * @throws \InvalidArgumentException Throws this exception if the post type is not organization.
	 */
	private function get_post_by_id( int $id ): WP_Post {
		$post = get_post( $id );
		if ( ! $post ) {
			throw new OrganizationNotFoundException();
		}
		if ( $this->post_type !== $post->post_type ) {
			throw new InvalidArgumentException( __( 'ID does not belong to a Organization Post', 'ld-organization' ) );
		}
		return $post;
	}

	/**
	 * Handles fetching the Organization based on the user ID.
	 *
	 * @param int    $id The user ID to search for.
	 * @param string $key The key to query for, defaults to 'organization_owners', can be either 'organization_owners', 'group_leader' or 'both'.
	 *
	 * @return WP_Post
	 * @throws \InvalidArgumentException If the key is not in the allowed ranges.
	 * @throws OrganizationNotFoundException Throws this exception if no organization is found.
	 */
	private function get_post_by_user_id( int $id, string $key = 'both' ): WP_Post {
		$args = array(
			'post_type'   => $this->post_type,
			'numberposts' => 1,
		);
		if ( 'both' === $key ) {
			$args['meta_query'] = array(
				'relation' => 'OR',
				array(
					'key'     => 'organization_owners',
					'value'   => '"' . $id . '"',
					'compare' => 'LIKE',
				),
				array(
					'key'     => 'organization_owners',
					'value'   => '"' . $id . '"',
					'compare' => 'LIKE',
				),
			);
		} elseif ( in_array( $key, array( 'organization_owners', 'group_leader' ), true ) ) {
			$args['meta_query'] = array(
				array(
					'key'     => $key,
					'value'   => '"' . $id . '"',
					'compare' => 'LIKE',
				),
			);
		} else {
			throw new \InvalidArgumentException( "The key needs to be either: 'organization_owners', 'group_leader' or 'both'" );
		}
		$posts = get_posts(
			$args
		);

		if ( empty( $posts ) ) {
			throw new OrganizationNotFoundException();
		}
		return $posts[0];
	}

	/**
	 * Sets the organization.
	 *
	 * @param WP_Post|null $organization The organization or null.
	 */
	final public function set_organization( ?WP_Post $organization ): void {
		$this->organization = $organization;
	}
}
