<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
// phpcs:disable Generic.Commenting.DocComment.MissingShort

namespace LD_Organization\Hooks;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use InvalidArgumentException;
use LD_Organization\Exceptions\OrganizationNotFoundException;
use LD_Organization\Organization;

/**
 * Handles calling the Organization for the current group and updating the Organizations licenses.
 *
 * @package LD_Organization\Hooks
 * @since 0.12.0 Added Class
 */
class OrganizationLicenseHooks extends AbstractHook {

	/**
	 * Checks if we are currently trying to redirect after having registered a user using Uncanny Toolkit
	 *
	 * @param string $location The current location that will be redirected to.
	 * @param int $status The current status of the redirect.
	 *
	 * @return string
	 * @see LearnDashGroupSignUp For more information on how uncanny toolkit does this.
	 */
	final public function add_failure_to_url( string $location, int $status ): string {

		// One of these should be defined and then we can be sure that we are currently on the registration page.
		if ( ! $this->is_uc_toolkit_signup() ) {
			return $location;
		}

		// We remove all the other as to not have any duplicate/misleading messages.
		return add_query_arg(
			array(
				'group_full' => true,
				'joined'     => false,
				'msg'        => false,
				'registered' => false,
			),
			$location
		);
	}

	/**
	 * Checks if we are currently in the signup process from Uncanny Toolkit.
	 *
	 * @return bool
	 */
	final public function is_uc_toolkit_signup(): bool {
		// phpcs:disable WordPress.Security.NonceVerification.Missing,WordPress.Security.NonceVerification.Recommended
		$is_register = isset( $_POST['uncanny_group_signup_register_nonce'] );
		$is_join     = isset( $_POST['uncanny_pro_toolkit_join_group_nonce'] );
		$might_be    = isset( $_REQUEST['gid'] ); // This one might be to "broad" but it should work as it is being used by UCToolkit.
		// phpcs:enable WordPress.Security.NonceVerification.Missing,WordPress.Security.NonceVerification.Recommended
		return $is_register || $is_join || $might_be;
	}

	/**
	 * Handles calling the correct function.
	 *
	 * @param int $user_id The ID of the user being added.
	 * @param int $group_id The affected group ID.
	 *
	 * @return void
	 */
	final public function add_group_access( int $user_id, int $group_id ): void {
		$this->check_licenses( $user_id, $group_id, false );
	}

	/**
	 * Handles checking if the group belongs to any organizations, and if so, updates the license information.
	 *
	 * @param int  $user_id The ID of the user being added.
	 * @param int  $group_id The affected group ID.
	 * @param bool $is_removing Is the user being removed from the group.
	 *
	 * @return void
	 */
	final public function check_licenses( int $user_id, int $group_id, bool $is_removing ): void {
		try {
			$organization_id = get_field( 'group_organization', $group_id );
			if ( empty( $organization_id ) ) {
				return;
			}
			$organization = Organization::with_id( $organization_id );

			if ( ! $organization->has_organization() ) {
				return;
			}

			$organization->recalculate_licenses();

		} catch ( OrganizationNotFoundException $e ) { // phpcs:ignore Generic.CodeAnalysis.EmptyStatement.DetectedCatch
			// No need to handle this as this can be run anywhere.
			return;
		} catch ( InvalidArgumentException $e ) {
			if ( $is_removing ) {
				return; // No need to handle this as this as we are removing from the group anyhow.
			}

			if ( current_user_can( 'manage_options' ) ) {
				return; // An admin can do what they pleases.
			}

			$removed = ld_update_group_access( $user_id, $group_id, true );
			if ( $removed ) {
				// We add this since we have removed a user from the group to signal to the frontend that we are full.
				add_filter( 'wp_redirect', array( $this, 'add_failure_to_url' ), 10, 2 );
			}
		}
	}

	/**
	 * Handles calling the correct function.
	 *
	 * @param int $user_id The ID of the user being added.
	 * @param int $group_id The affected group ID.
	 *
	 * @return void
	 */
	final public function remove_group_access( int $user_id, int $group_id ): void {
		$this->check_licenses( $user_id, $group_id, true );
	}

	/**
	 * @inheritDoc
	 */
	public function get_actions(): array {
		return array(
			array(
				'hook'     => 'ld_added_group_access',
				'callable' => array( $this, 'add_group_access' ),
				'priority' => 10,
				'num_args' => 2,
			),
			array(
				'hook'     => 'ld_removed_group_access',
				'callable' => array( $this, 'remove_group_access' ),
				'priority' => 10,
				'num_args' => 2,
			),
		);
	}

	/**
	 * @inheritDoc
	 */
	public function get_filters(): array {
		return array();
	}
}
