<?php
/**
 * Group helper functions and other functionality.
 *
 * @package LD_Organization
 * @since 0.4.0
 */

namespace LD_Organization;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use InvalidArgumentException;
use LD_Organization\Exceptions\OrganizationNotFoundException;
use LD_Organization\Exceptions\OrganizationOutOfLicensesException;
use WP_Post;
use WP_REST_Request;

/**
 * Handles updating the group if a group is created via the REST API.
 *
 * @param WP_Post         $post The WP_Post object.
 * @param WP_REST_Request $request The REST request object.
 * @param bool            $creating If the post is being created.
 *
 * @return void
 */
function handle_rest_group_creation( WP_Post $post, WP_REST_Request $request, bool $creating ) {
	if ( ! $creating || ! is_user_logged_in() ) {
		return;
	}
	try {
		$user_id      = get_current_user_id();
		$organization = Organization::by_user_id( $user_id );
		$organization->initialize_group( $post->ID, array( $user_id ) );

	} catch ( InvalidArgumentException | OrganizationOutOfLicensesException | OrganizationNotFoundException $e ) {
		return;
	}
}

/**
 * Handles building a signed enrollment url for a group.
 *
 * @param int $group_id The group ID to build a url for.
 *
 * @since 0.11.6 Uses the new url method from uncanny toolkit.
 * @return string
 */
function get_signed_enrollment_url( int $group_id ): string {
	if ( ! learndash_is_group_post( $group_id ) ) {
		return '';
	}
	$post           = get_post( $group_id );
	$enrollment_url = site_url( 'sign-up/' . $post->post_name . '/' );
	// Creates the "group key, from the post ID (Uncanny toolkit code).
	$group_key = crypt( (string) $group_id, 'uncanny-group' );
	// Fixing $_GET string from . (dot) & space to _ ( underscore ).
	$group_key      = str_replace( array( ' ', '.', '[', '-' ), '_', $group_key );
	$enrollment_url = add_query_arg( array( 'gid' => $group_id ), $enrollment_url );

	return esc_url( $enrollment_url ) . '&' . rawurlencode( $group_key );
}


/**
 * @param int $user_id The user ID of the user being checked.
 * @param int $group_id The group ID.
 *
 * @return bool
 */
function can_user_be_deleted_from_group( int $user_id, int $group_id ): bool {
	// Check if the user is part of the group.
	if ( ! ( learndash_is_user_in_group( $user_id, $group_id ) ) ) {
		return false;
	}

	$can_delete = learndash_get_user_group_status( $group_id, $user_id, true ) === 'not-started';
	// Prevent group leaders/admins to be deleted.
	if ( $can_delete ) {
		$can_delete = ! ( learndash_is_admin_user( $user_id ) || learndash_is_group_leader_user( $user_id ) );
	}

	return $can_delete;
}
