<?php
/**
 * Handles registering the RestAPI endpoints for handling organizations.
 *
 * @package LD_Organization
 */

namespace LD_Organization;

use InvalidArgumentException;
use LDLMS_Model_Course;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * RestAPI class, when initialized will register the restAPI.
 *
 * @package LD_Organization
 * @since 0.2.0
 */
class RestAPI {

	/**
	 * The RestAPI namespace.
	 *
	 * @var string
	 */
	private string $namespace = 'ld-organization';

	/**
	 * The RestAPI version.
	 *
	 * @var string
	 */
	private string $version = 'v1';

	/**
	 * Registers all required functionality and endpoints.
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_endpoints' ) );
	}

	/**
	 * Registers all the endpoint and their parameters.
	 *
	 * @return void
	 */
	final public function register_endpoints(): void {
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/licenses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_licenses' ),
				'permission_callback' => array( $this, 'is_owner' ),
				'args'                => $this->validate_id_argument(),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_organization' ),
				'permission_callback' => array( $this, 'is_logged_in' ),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'group/(?P<group_id>\d+)/course/(?P<course_id>\d+)/information',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_extended_course_information' ),
				'permission_callback' => array( $this, 'is_group_leader_of_post' ),
				'args'                => array(
					'course_id' => $this->is_valid_ld_course_id(),
					'group_id'  => $this->is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_users_groups' ),
				'permission_callback' => array( $this, 'is_logged_in' ),
				'args'                => array(
					'role' => array(
						'validate_callback' => function( $param, $request, $key ) {
							return is_string( $param ) && in_array( (string) $param, $this->get_valid_roles_array(), true );
						},
					),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/courses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_groups_course_ids' ),
				'permission_callback' => array( $this, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id' => $this->is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/new_courses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_organization_allowed_courses' ),
				'permission_callback' => array( $this, 'is_group_leader' ),
				'args'                => array(
					'id'       => $this->is_valid_org_id(),
					'group_id' => $this->is_valid_ld_group_id(),
				),
			)
		);

		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/courses',
			array(
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => array( $this, 'add_course_to_organization_group' ),
				'permission_callback' => array( $this, 'is_group_leader' ),
				'args'                => array(
					'id'        => $this->is_valid_org_id(),
					'group_id'  => $this->is_valid_ld_group_id(),
					'course_id' => $this->is_valid_ld_course_id(),
				),
			)
		);
	}

	/**
	 * Fetches the requested organization and their allowed courses, if a school, otherwise it returns all courses.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 * @since 0.5.0
	 */
	final public function get_organization_allowed_courses( WP_REST_Request $request ): WP_REST_Response {
		try {
			$organization = Organization::with_id( $request->get_param( 'id' ) );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$course_ids       = $organization->get_allowed_courses_ids();
			$group_id         = $request->get_param( 'group_id' );
			$group_course_ids = learndash_get_groups_courses_ids( get_current_user_id(), array( $group_id ) );
			$final_course_ids = array_filter(
				$course_ids,
				static function ( $course_id ) use ( $group_course_ids ) {
					return ! in_array( $course_id, $group_course_ids, true );
				}
			);

			$response = new WP_REST_Response(
				array(
					'success'    => true,
					'course_ids' => array_values( $final_course_ids ),
				),
				200
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				404
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Handles fetching the organization based on the current user.
	 *
	 * @param WP_REST_Request $request The current request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_organization( WP_REST_Request $request ): WP_REST_Response {
		try {
			$user_id      = get_current_user_id();
			$organization = Organization::by_user_id( $user_id );
			$post         = $organization->get_organization();
			if ( null !== $post ) {
				$organization_data = array(
					'success' => true,
					'data'    => array(
						'id'   => $post->ID,
						'name' => $post->post_title,
						'slug' => $post->post_name,
						'type' => $organization->get_type(),
					),
				);
			} else {
				throw new OrganizationNotFoundException();
			}
			return new WP_REST_Response( $organization_data, 200 );
		} catch ( OrganizationNotFoundException $e ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found.',
				),
				404
			);
		}
	}

	/**
	 * Returns the requested organizations licenses.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @uses \LD_Organization\Organization::with_id(),\LD_Organization\Organization::get_licenses(),\LD_Organization\Organization::get_used_licenses()
	 * @return WP_REST_Response
	 */
	final public function get_licenses( WP_REST_Request $request ): WP_REST_Response {
		try {
			$id           = $request->get_param( 'id' );
			$organization = Organization::with_id( $id );

			$used_licenses      = $organization->get_used_licenses();
			$total_licenses     = $organization->get_licenses();
			$remaining_licenses = $organization->get_remaining_licenses();

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => array(
						'used_licenses'      => $used_licenses,
						'total_licenses'     => $total_licenses,
						'remaining_licenses' => $remaining_licenses,
					),
					'message' => 'Successfully fetched licenses for organization' . $id,
				)
			);

		} catch ( InvalidArgumentException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $exception->getMessage(),
				),
				400
			);
		} catch ( OrganizationNotFoundException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				404
			);
		}
		return rest_ensure_response( $response );
	}

	/**
	 * Creates an extended information for a course based on the requesting group.
	 *
	 * @param WP_REST_Request $request The request object of the current WP_REST_Request instance.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_extended_course_information( WP_REST_Request $request ): WP_REST_Response {
		$course_id = (int) $request->get_param( 'course_id' );
		$group_id  = (int) $request->get_param( 'group_id' );

		$transient_name = 'extended_information_' . $group_id . '_' . $course_id;

		if ( false !== ( $value = LDOrgTransientManager::get( $transient_name ) ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $value,
				),
				200
			);
		}

		/** The course object @var LDLMS_Model_Course|null $course */
		$course = \LDLMS_Factory_Post::course( $course_id );
		if ( null === $course ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => __(
						'Course not found',
						'ld-organization'
					),
				),
				400
			);
		}

		$groups_ids = learndash_get_course_groups( $course_id );

		if ( ! in_array( $group_id, $groups_ids, true ) ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'This group does not have access to this course', 'ld-organization' ),
				),
				403
			);
		}
		$extended_information = array();

		$extended_information['course_progress'] = calculate_groups_course_progress( $group_id, $course_id );
		$image_url                               = get_the_post_thumbnail_url( $course_id );
		if ( '' === $image_url || false === $image_url ) {
			$image_url = get_default_image_url( 'course', 'medium' );
		}
		$extended_information['image_url'] = $image_url;

		if ( ! empty( $extended_information ) ) {
			LDOrgTransientManager::set( $transient_name, $extended_information, 5 * MINUTE_IN_SECONDS );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $extended_information,
			),
			200
		);

	}

	/**
	 * Gets the requesting users groups (depending on role param) and returns their ids.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_users_groups( WP_REST_Request $request ): WP_REST_Response {
		$type            = $request->get_param( 'role' );
		$current_user_id = get_current_user_id();
		// TODO: Maybe implement the other roles, if needed.
		if ( 'group_leader' === $type ) {
			$group_ids = learndash_get_administrators_group_ids( $current_user_id );
			$response  = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $group_ids,
				)
			);
		}

		if ( ! isset( $response ) ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'No role specified',
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Returns a list of course IDs for the requested group.
	 *
	 * @param WP_REST_Request $request The current request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_groups_course_ids( WP_REST_Request $request ): WP_REST_Response {
		$group_id    = $request->get_param( 'group_id' );
		$courses_ids = learndash_get_groups_courses_ids( get_current_user_id(), array( $group_id ) );

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $courses_ids,
			)
		);
	}

	/**
	 * Handles adding course to organization.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 * @since 0.6.0
	 */
	final public function add_course_to_organization_group( WP_REST_Request $request ): WP_REST_Response {
		$group_id      = $request->get_param( 'group_id' );
		$org_id        = $request->get_param( 'id' );
		$new_course_id = $request->get_param( 'course_id' );

		try {
			$organization = Organization::with_id( $org_id );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$allowed_courses = $organization->get_allowed_courses_ids();
			if ( ! in_array( $new_course_id, $allowed_courses, true ) ) {
				$response = new WP_REST_Response(
					array(
						'success' => false,
						'message' => 'This course is not allowed to be added to this organization',
					),
					403
				);
			} else {

				$result = ld_update_course_group_access( $new_course_id, $group_id, false );
				if ( false === $result ) {
					$response = new WP_REST_Response(
						array(
							'success' => false,
							'message' => 'Failed to add course to group',
						),
						500
					);
				} else {
					$response = new WP_REST_Response(
						array(
							'success' => true,
							'message' => 'Added course to group',
						),
						201
					);
				}
			}
		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				404
			);
		}

		return rest_ensure_response( $response );

	}

	/**
	 * PERMISSION CALLBACKS
	 */

	/**
	 * Checks if the user is logged in.
	 *
	 * @return bool
	 */
	final public function is_logged_in(): bool {
		return is_user_logged_in();
	}

	/**
	 * Handles checking if the user is logged in and group leader for the requested course.
	 *
	 * @param WP_REST_Request $request The current WP_REST_Request object.
	 *
	 * @return bool
	 */
	final public function is_group_leader_of_post( WP_REST_Request $request ): bool {
		if ( ! is_user_logged_in() ) {
			return false;
		}

		$current_user = wp_get_current_user();

		if ( null === $current_user ) {
			return false;
		}

		// Current user is an admin user, allow all.
		if ( current_user_can( 'manage_options' ) ) {
			return true;
		}

		// If we pass the WP checks, we check if the user is a group leader and is a part of the group administrators.
		$admins = learndash_get_groups_administrator_ids( $request->get_param( 'group_id' ) );

		if ( ! in_array( $current_user->ID, $admins, true ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Checks if the user calling the API is an owner of the organization being called.
	 *
	 * @param WP_REST_Request $request The request object for the current request.
	 * @param string          $param_key The parameter key to check.
	 *
	 * @return bool
	 * @uses \LD_Organization\RestAPI::has_access()
	 */
	final public function is_owner( WP_REST_Request $request, string $param_key = 'id' ): bool {
		return $this->has_access( $request, 'organization_owners', $param_key );
	}

	/**
	 * Checks if the user calling the API is an owner of the organization being called.
	 *
	 * @param WP_REST_Request $request The request object for the current request.
	 * @param string          $param_key The parameter key to check for organization.
	 *
	 * @return bool
	 * @uses \LD_Organization\RestAPI::has_access()
	 */
	final public function is_group_leader( WP_REST_Request $request, string $param_key = 'id' ): bool {
		return $this->has_access( $request, 'group_leaders', $param_key );
	}

	/**
	 * Handles checking if the organization is owned by the authenticated user.
	 *
	 * @param WP_REST_Request $request The request with a required parameter of ID.
	 * @param string          $role The role to check against.
	 * @param string          $param_key Optionally the key of the parameter to check.
	 *
	 * @return bool
	 * @uses \LD_Organization\Organization::with_id(),\LD_Organization\Organization::has_user()
	 */
	final public function has_access( WP_REST_Request $request, string $role = 'both', string $param_key = 'id' ): bool {
		if ( ! is_user_logged_in() ) {
			return false;
		}
		$id = $request->get_param( $param_key );
		if ( ! isset( $id ) ) {
			return false;
		}
		try {
			$organization = Organization::with_id( $id );
			$current_user = wp_get_current_user();
			if ( isset( $current_user ) && $organization->has_user( $current_user->ID, $role ) ) {
				return true;
			}
		} catch ( OrganizationNotFoundException | InvalidArgumentException $exception ) {
			return false;
		}
		return false;
	}

	/**
	 *  VALIDATION CALLBACKS.
	 */

	/**
	 * The default validate_callback if the endpoint has an ID in it.
	 *
	 * @return array
	 */
	final public function validate_id_argument(): array {
		return array(
			'id' => array(
				'validate_callback' => function( $param, $request, $key ) {
					return is_numeric( $param );
				},
			),
		);
	}

	/**
	 * Checks if the parameter is a valid group_id.
	 *
	 * @return array
	 */
	final public function is_valid_ld_group_id(): array {
		return array(
			'validate_callback' => function( $param, $request, $key ) {
				return is_numeric( $param ) && learndash_is_group_post( $param );
			},
		);
	}

	/**
	 * Checks if the parameter is a valid group post ID.
	 *
	 * @return array
	 */
	final public function is_valid_ld_course_id(): array {
		return array(
			'validate_callback' => function( $param, $request, $key ) {
				return is_numeric( $param ) && learndash_is_course_post( $param );
			},
		);
	}

	/**
	 * Checks if the parameter is a valid organization ID.
	 *
	 * @return array
	 */
	final public function is_valid_org_id(): array {
		return array(
			'validate_callback' => function( $param, $request, $key ) {
				return is_numeric( $param ) && is_valid_organization_post( $param );
			},
		);
	}

	/**
	 * GETTERS
	 */

	/**
	 * Gets the namespace string for the Rest API.
	 *
	 * @return string
	 */
	final public function get_namespace(): string {
		return $this->namespace . '/' . $this->version;
	}

	/**
	 * Returns the array of valid roles that can be requested.
	 *
	 * @return string[]
	 */
	private function get_valid_roles_array(): array {
		return array( 'group_leader', 'organization_owner' );
	}
}
