<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
/**
 * AbstractHook class is an abstract class that provides some sane defaults for registering hooks.
 *
 * @package LD_Organization\Hooks
 */

namespace LD_Organization\Hooks;

use InvalidArgumentException;

/**
 * Handles registering hooks.
 *
 * @since 0.11.0 Added abstract hook class.
 * @package LD_Organization\Hooks
 */
abstract class AbstractHook {
	/**
	 * All actions are defined here in the format:
	 * array( 'hook' => $hook_name, callable' => $callable, 'priority' => $priority, 'num_args' => $num_args )
	 *
	 * @var array
	 */
	protected array $actions = array();

	/**
	 * All filters are defined here in the format:
	 * array( 'hook' => $hook_name, callable' => $callable, 'priority' => $priority, 'num_args' => $num_args )
	 *
	 * @var array
	 */
	protected array $filters = array();

	/**
	 * Registers actions and hooks defined on the object.
	 *
	 * @param array $actions An array of actions to register.
	 * @param array $filters An array of filters to register.
	 */
	public function __construct( array $actions = array(), array $filters = array() ) {
		$this->actions = $actions;
		$this->filters = $filters;
		$this->register_actions();
		$this->register_filters();
	}

	/**
	 * Handles registering the actions.
	 *
	 * @throws InvalidArgumentException If the action hooks are not defined correctly.
	 * @see AbstractHook::actions
	 * @return void
	 */
	private function register_actions(): void {
		foreach ( $this->actions as $action ) {
			if ( ! isset( $action['hook'], $action['callable'] ) ) {
				throw new InvalidArgumentException( "Actions are required to have the attributes 'hook' and 'callable' set." );
			}
			add_action( $action['hook'], $action['callable'], $action['priority'] ?? 10, $action['num_args'] ?? 1 );
		}
	}

	/**
	 * Handles registering the filters.
	 *
	 * @throws InvalidArgumentException If the filter hooks are not defined correctly.
	 * @see AbstractHook::filters
	 * @return void
	 */
	private function register_filters(): void {
		foreach ( $this->filters as $filter ) {
			if ( ! isset( $filter['hook'], $filter['callable'] ) ) {
				throw new InvalidArgumentException( "Actions are required to have the attributes 'hook' and 'callable' set." );
			}
			add_filter( $filter['hook'], $filter['callable'], $filter['priority'] ?? 10, $filter['num_args'] ?? 1 );
		}
	}

}
