<?php
/**
 * Organization class
 *
 * @package LD_Organization
 */

namespace LD_Organization;

// Require all the exceptions to allow for usage.
require_once __DIR__ . '/exceptions/class-organizationnotfoundexception.php';

use InvalidArgumentException;
use WP_Post;
use WP_User;

/**
 * The Organization class is responsible fetching and checking different attributes revolving around organizations,
 * can and should be used when directly communicating with Learndash.
 *
 * @package LD_Organization
 * @since 0.1.0
 */
class Organization {

	/**
	 * The post type slug to use.
	 * Default: "organization".
	 *
	 * @var string
	 */
	private string $post_type = POST_TYPE_NAME;

	/**
	 * The organization being handled.
	 *
	 * @var WP_Post|null
	 */
	private ?WP_Post $organization;

	/**
	 * TODO: Needs a constructor?
	 */
	public function __construct() {
	}

	/**
	 * Handles initializing Organization class with ID.
	 *
	 * @param int $id The ID of the organization to retrieve.
	 *
	 * @return Organization
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws InvalidArgumentException If the ID belongs to a non-organization post.
	 */
	public static function with_id( int $id ): Organization {
		$instance = new self();
		$post     = $instance->get_post_by_id( $id );
		$instance->set_organization( $post );
		return $instance;
	}

	/**
	 * Handles initializing Organization class based on a user ID.
	 *
	 * @param int    $id The ID of a user belonging to the organization to retrieve.
	 * @param string $key The meta key to query on, uses Organization::get_post_by_user_id so the accepted values are the same as that.
	 *
	 * @return Organization
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws InvalidArgumentException If the ID belongs to a non-organization post or the passed key is invalid.
	 *
	 * @uses Organization::get_post_by_user_id()
	 */
	public static function by_user_id( int $id, string $key = 'both' ): Organization {
		$instance = new self();
		$post     = $instance->get_post_by_user_id( $id, $key );
		$instance->set_organization( $post );
		return $instance;
	}

	/**
	 * Handles initializing Organization class based on a user ID.
	 *
	 * @param string $email The email of the user belonging to an organization.
	 * @param string $key The meta key to query on, uses Organization::get_post_by_user_id so the accepted values are the same as that.
	 *
	 * @return Organization
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws InvalidArgumentException If the ID belongs to a non-organization post or the passed key is invalid.
	 *
	 * @uses Organization::get_post_by_user_email()
	 */
	public static function by_user_email( string $email, string $key = 'both' ): Organization {
		$instance = new self();
		$post     = $instance->get_post_by_user_email( $email, $key );
		$instance->set_organization( $post );
		return $instance;
	}

	/**
	 * Handles initializing Organization class based on the user ID.
	 *
	 * @param string $email The email to lookup user by.
	 * @param string $key The meta key to query on, uses Organization::get_post_by_user_id so the accepted values are the same as that.
	 *
	 * @return WP_Post
	 * @throws OrganizationNotFoundException If the organization is not found.
	 * @throws InvalidArgumentException If a user is not found, because the email is not found.
	 *
	 * @uses Organization::get_post_by_user_id()
	 */
	private function get_post_by_user_email( string $email, string $key = 'both' ): WP_Post {
		$user = get_user_by( 'email', $email );
		if ( ! $user ) {
			throw new InvalidArgumentException( 'No user with the provided email found' );
		}
		return $this->get_post_by_user_id( $user->ID, $key );
	}

	/**
	 * Handles fetching the post by ID.
	 *
	 * @param int $id The Post id.
	 *
	 * @return WP_Post
	 * @throws OrganizationNotFoundException Throws this exception if a Organization cannot be found.
	 * @throws InvalidArgumentException Throws this exception if the post type is not organization.
	 */
	private function get_post_by_id( int $id ): WP_Post {
		$post = get_post( $id );
		if ( ! $post ) {
			throw new OrganizationNotFoundException();
		}
		if ( $this->post_type !== $post->post_type ) {
			throw new InvalidArgumentException( __( 'ID does not belong to a Organization Post', 'ld-organization' ) );
		}
		return $post;
	}

	/**
	 * Handles fetching the Organization based on the user ID.
	 *
	 * @param int    $id The user ID to search for.
	 * @param string $key The key to query for, defaults to 'organization_owners', can be either 'organization_owners', 'group_leaders' or 'both'.
	 *
	 * @return WP_Post
	 * @throws InvalidArgumentException If the key is not in the allowed ranges.
	 * @throws OrganizationNotFoundException Throws this exception if no organization is found.
	 */
	private function get_post_by_user_id( int $id, string $key = 'both' ): WP_Post {
		$args = array(
			'post_type'   => $this->post_type,
			'numberposts' => 1,
		);
		if ( 'both' === $key ) {
			$args['meta_query'] = array(
				'relation' => 'OR',
				array(
					'key'     => 'organization_owners',
					'value'   => '"' . $id . '"',
					'compare' => 'LIKE',
				),
				array(
					'key'     => 'group_leaders',
					'value'   => '"' . $id . '"',
					'compare' => 'LIKE',
				),
			);
		} elseif ( in_array( $key, array( 'organization_owners', 'group_leaders' ), true ) ) {
			$args['meta_query'] = array(
				array(
					'key'     => $key,
					'value'   => '"' . $id . '"',
					'compare' => 'LIKE',
				),
			);
		} else {
			throw new InvalidArgumentException( "The key needs to be either: 'organization_owners', 'group_leaders' or 'both'" );
		}
		$posts = get_posts(
			$args
		);

		if ( empty( $posts ) ) {
			throw new OrganizationNotFoundException();
		}
		return $posts[0];
	}

	/**
	 * Set the used licenses to a fixed number
	 *
	 * This method does a bunch of checks to see if we are allowed to do this.
	 *
	 * @param int $used_licenses The number of licenses to set as used.
	 *
	 * @return int|bool
	 * @throws OrganizationNotFoundException In case there is no organization defined on the object.
	 * @throws InvalidArgumentException Will be thrown if the number of licenses exceeds the allowed licenses on the organization.
	 */
	final public function set_used_licenses( int $used_licenses ): int {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		$allowed_licenses = $this->get_licenses();
		if ( $used_licenses > $allowed_licenses ) {
			throw new InvalidArgumentException( 'Used licenses cannot be higher than available licenses' );
		}
		update_post_meta( $this->organization->ID, USED_LICENSES_META_KEY, $used_licenses );

		return $used_licenses;
	}

	/**
	 * Increases the used licenses by the specified amount
	 *
	 * This method does a bunch of checks to see if we are allowed to do this.
	 *
	 * @param int $amount The amount of licenses to "use".
	 *
	 * @return int|bool
	 * @throws OrganizationNotFoundException In case there is no organization defined on the object.
	 * @throws InvalidArgumentException If the amount would make the used_licenses larger than the maximum allowed licenses.
	 */
	final public function increment_used_licenses( int $amount ): int {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		$allowed_licenses      = $this->get_licenses();
		$current_used_licenses = $this->get_used_licenses();
		if ( ( $amount + $current_used_licenses ) > $allowed_licenses ) {
			throw new InvalidArgumentException( 'Used licenses cannot be higher than available licenses' );
		}
		update_post_meta( $this->organization->ID, USED_LICENSES_META_KEY, ( $amount + $current_used_licenses ) );

		return ( $amount + $current_used_licenses );
	}

	/**
	 * Checks if the user is part of the owners of the organization.
	 *
	 * @param int    $user_id The user ID to check against.
	 * @param string $role The role to check against, can be either 'organization_owners' or 'group_leaders' or 'both'.
	 *
	 * @return bool
	 * @throws OrganizationNotFoundException If this object has no organization is defined.
	 * @throws InvalidArgumentException If the role is not a valid value.
	 */
	final public function has_user( int $user_id, string $role = 'both' ): bool {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		if ( ! in_array( $role, array( 'both', 'organization_owners', 'group_leaders' ), true ) ) {
			throw new InvalidArgumentException( "Role has to be either 'organization_owners' or 'group_leaders' or 'both'" );
		}

		// Fetch both if that is required and merge the arrays.
		if ( 'both' === $role ) {
			$owners        = get_field( 'organization_owners', $this->organization->ID );
			$group_leaders = get_field( 'group_leaders', $this->organization->ID );
			$users         = array_merge( $owners, $group_leaders );
		} else {
			$users = get_field( $role, $this->organization->ID );
		}

		if ( ! is_array( $users ) || empty( $users ) ) {
			return false;
		}

		$filtered_arr = array_filter(
			$users,
			static function( WP_User $user ) use ( $user_id ) {
				return $user_id === $user->ID;
			}
		);

		return ! empty( $filtered_arr );
	}

	/**
	 * Fetches the number of used licenses from the database.
	 *
	 * @return int
	 * @throws OrganizationNotFoundException If the organization is not defined on the object.
	 */
	final public function get_used_licenses(): int {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}
		$number_used_licenses = (int) get_post_meta( $this->organization->ID, USED_LICENSES_META_KEY, true );
		if ( empty( $number_used_licenses ) && 0 !== $number_used_licenses ) {
			return 0;
		}
		return $number_used_licenses;
	}

	/**
	 * Handles fetching the amount of allowed licenses for this organization.
	 *
	 * @return int
	 * @throws OrganizationNotFoundException If the organization is not defined on the object.
	 */
	final public function get_licenses(): int {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		return (int) get_field( 'licenses', $this->organization->ID, true );
	}

	/**
	 * Returns the remaining licenses of this organization.
	 *
	 * @return int
	 * @throws OrganizationNotFoundException Thrown if the organization is not defined on the object.
	 */
	final public function get_remaining_licenses(): int {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		return $this->get_licenses() - $this->get_used_licenses();
	}

	/**
	 * Returns whether this class has an organization or not.
	 *
	 * @return bool
	 */
	final public function has_organization(): bool {
		return isset( $this->organization );
	}

	/**
	 * Sets the organization.
	 *
	 * @param WP_Post|null $organization The organization or null.
	 */
	final public function set_organization( ?WP_Post $organization ): void {
		$this->organization = $organization;
	}

	/**
	 * Returns the organization WP_Post object.
	 *
	 * @return WP_Post|null
	 */
	final public function get_organization(): ?WP_Post {
		return $this->organization;
	}

	/**
	 * Returns the organizations type as a string.
	 *
	 * @return string
	 * @throws OrganizationNotFoundException Thrown if the organization does not exists on this object.
	 */
	final public function get_type(): string {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		return get_field( 'organization_type', $this->organization->ID, true );
	}

	/**
	 * Checks whether this organization is of type school.
	 *
	 * @return bool
	 * @throws OrganizationNotFoundException Thrown if the organization does not exists on this object.
	 * @uses Organization::get_type()
	 * @since 0.3.0
	 */
	final public function is_school(): bool {
		if ( ! $this->has_organization() ) {
			throw new OrganizationNotFoundException();
		}

		return $this->get_type() === 'school';
	}
}
