<?php
/**
 * The notification class.
 *
 * @package LD_Organization
 */

namespace LD_Organization;

require_once __DIR__ . '/exceptions/class-typenotfoundexception.php';

/**
 * Notification class, handles sending off notifications.
 *
 * @package LD_Organization
 */
class Notification {

	private $type         = '';
	protected $id         = 0;
	protected $to_email   = '';
	protected $to_name    = '';
	protected $from_email = '';
	protected $from_name  = '';
	protected $title      = '';
	protected $content    = '';
	protected $data       = array();
	protected $user       = 0;
	private int $user_id;

	/**
	 * Construct a Notification sender.
	 *
	 * @param string|null $type The type, should be the slug of a notification post-type.
	 *
	 * @throws TypeNotFoundException This is thrown if the type is set and a post is not found.
	 */
	public function __construct( string $type = null ) {
		$this->set_to( get_bloginfo( 'admin_email' ) );
		if ( ! empty( $type ) ) {
			$this->set_type( $type );
		}
		$user = wp_get_current_user();
		if ( $user ) {
			$this->user_id = $user->ID;

			$this->set_data(
				array(
					'user_id'         => $user->ID,
					'user_login'      => $user->user_login,
					'user_name'       => $user->display_name,
					'user_first_name' => $user->first_name,
					'user_last_name'  => $user->last_name,
					'user_email'      => $user->user_email,
					'user_nickname'   => $user->nickname,
					'user_phone'      => $user->billing_phone,
					'user_address'    => $user->billing_address_1,
					'user_city'       => $user->billing_city,
					'user_postcode'   => $user->billing_postcode,
				)
			);
		}

	}

	/**
	 * Sets the type, exceptions should be handled.
	 *
	 * @param string $type the "type" of notification to send, which is the slug of post-type notification.
	 *
	 * @throws TypeNotFoundException This exception is thrown if the type is not found.
	 */
	final public function set_type( string $type ): void {
		$post = get_page_by_path( $type, OBJECT, 'notification' );
		if ( ! empty( $post ) ) {
			$this->type    = $type;
			$this->id      = $post->ID;
			$this->title   = $post->post_title;
			$this->content = $post->post_content;
		} else {
			throw new TypeNotFoundException();
		}
	}

	/**
	 * Setter for data.
	 *
	 * @param array $data The data array.
	 *
	 * @return void
	 */
	final public function set_data( array $data ): void {
		foreach ( $data as $key => $value ) {
			$this->data[ $key ] = $value;
		}
	}

	/**
	 * Setter for the to recipient.
	 *
	 * @param string $recipient the recipient.
	 *
	 * @return void
	 */
	final public function set_to( string $recipient ): void {
		if ( is_numeric( $recipient ) ) {
			$user = get_user_by( 'ID', $recipient );
			if ( $user ) {
				$this->to_email = $user->user_email;
				$this->to_name  = $user->display_name;
			}
		} else {
			$this->to_email = $recipient;
		}
	}

	/**
	 * Returns a list of the recipients for the current type.
	 *
	 * @return array
	 */
	final public function get_recipients(): array {
		$recipients = array();
		foreach ( get_field( 'notifications', $this->id ) as $recipient ) {
			$recipients[] = $recipient['recipient'];
		}

		return $recipients;
	}

	/**
	 * Setter for the from sender.
	 *
	 * @param string $sender the sender.
	 *
	 * @return void
	 */
	final public function set_from( string $sender ): void {
		$this->from_email = $sender;
	}

	/**
	 * Getter for the to recipient.
	 *
	 * @return string
	 */
	final public function get_to(): string {
		return $this->build_mail_string( $this->to_email, $this->to_name );
	}

	/**
	 * Getter for the from sender.
	 *
	 * @return string
	 */
	final public function get_from(): string {
		return $this->build_mail_string( $this->from_email, $this->from_name );
	}

	/**
	 * Handles the building of a correct mail string.
	 *
	 * @param string $email The email.
	 * @param string $name The name.
	 *
	 * @return string
	 */
	final protected function build_mail_string( string $email, string $name = '' ): string {
		$address = $email;
		if ( ! empty( $name ) && ! empty( $email ) ) {
			$address = "$name <$email>";
		}

		return $address;
	}

	/**
	 * Handles sending the email.
	 *
	 * @param array $to all the recipients.
	 *
	 * @return bool
	 */
	public function send( array $to = array() ): bool {
		if ( empty( $this->id ) ) {
			return false;
		}
		$subject = get_bloginfo( 'name' ) . ' - ' . $this->parse( $this->title );
		$message = $this->parse( $this->content );

		$comment = array(
			'comment_post_ID'  => $this->id,
			'comment_approved' => 0,
			'comment_type'     => 'notification',
			'user_id'          => $this->user_id,
			'comment_content'  => "$subject\n\n$message",
		);
		wp_insert_comment( $comment );
		if ( empty( $to ) ) {
			$to = array( $this->get_to() );
		}
		$success = false;

		// We exit early if WP_DEBUG is true.
		if ( WP_DEBUG ) {
			return true;
		}

		if ( ! empty( $to ) ) {
			$success = true;
			foreach ( $to as $send_to ) {
				if ( ! wp_mail( $send_to, mb_encode_mimeheader( $subject, 'UTF-8', 'Q' ), $this->wrapper( $message ), $this->build_headers() ) ) {
					$success = false;
				}
			}
		}

		return $success;
	}

	/**
	 * Handles building the correct headers for the email.
	 *
	 * @return string[]
	 */
	final protected function build_headers(): array {
		$headers = array( 'Content-Type: text/html; charset=UTF-8' );
		if ( ! empty( $this->from_email ) ) {
			$headers[] = 'From: ' . $this->get_from();
		}

		return $headers;
	}

	/**
	 * Handles correctly parsing the text for template {{ }} values and setting the values.
	 *
	 * @param string $text The text to parse.
	 *
	 * @return array|string|string[]
	 */
	final protected function parse( string $text ) {
		if ( preg_match_all( '/\{\{ *([^} ]*) *\}\}/', $text, $matches, PREG_SET_ORDER ) ) {
			foreach ( $matches as $match ) {
				$value = $this->data[ $match[1] ] ?? '';
				if ( is_array( $value ) ) {
					$new = '';
					foreach ( $value as $k => $v ) {
						$new .= "$k: $v\n";
					}
					$value = $new;
				}

				$text = str_replace( $match[0], $value, $text );
			}
		}

		return $text;
	}

	/**
	 * Handles wrapping the final email in a correct HTML string.
	 *
	 * @param string $text The text to insert into the HTML.
	 *
	 * @return string
	 */
	private function wrapper( string $text ): string {
		return '<!doctype html><html lang="en"><head><meta charset="utf-8"></head><body>' . wpautop( $text ) . '</body></html>';
	}
}
