<?php
/**
 * Registering the organization post type and all required components.
 *
 * @package LD_Organization
 */

namespace LD_Organization;

/**
 * Registers the `organization` post type.
 */
function organization_init() {
	register_post_type(
		POST_TYPE_NAME,
		array(
			'labels'                => array(
				'name'                  => __( 'Organizations', 'ld-organization' ),
				'singular_name'         => __( 'Organization', 'ld-organization' ),
				'all_items'             => __( 'All Organizations', 'ld-organization' ),
				'archives'              => __( 'Organization Archives', 'ld-organization' ),
				'attributes'            => __( 'Organization Attributes', 'ld-organization' ),
				'insert_into_item'      => __( 'Insert into organization', 'ld-organization' ),
				'uploaded_to_this_item' => __( 'Uploaded to this organization', 'ld-organization' ),
				'featured_image'        => _x( 'Featured Image', 'organization', 'ld-organization' ),
				'set_featured_image'    => _x( 'Set featured image', 'organization', 'ld-organization' ),
				'remove_featured_image' => _x( 'Remove featured image', 'organization', 'ld-organization' ),
				'use_featured_image'    => _x( 'Use as featured image', 'organization', 'ld-organization' ),
				'filter_items_list'     => __( 'Filter organizations list', 'ld-organization' ),
				'items_list_navigation' => __( 'Organizations list navigation', 'ld-organization' ),
				'items_list'            => __( 'Organizations list', 'ld-organization' ),
				'new_item'              => __( 'New Organization', 'ld-organization' ),
				'add_new'               => __( 'Add New', 'ld-organization' ),
				'add_new_item'          => __( 'Add New Organization', 'ld-organization' ),
				'edit_item'             => __( 'Edit Organization', 'ld-organization' ),
				'view_item'             => __( 'View Organization', 'ld-organization' ),
				'view_items'            => __( 'View Organizations', 'ld-organization' ),
				'search_items'          => __( 'Search organizations', 'ld-organization' ),
				'not_found'             => __( 'No organizations found', 'ld-organization' ),
				'not_found_in_trash'    => __( 'No organizations found in trash', 'ld-organization' ),
				'parent_item_colon'     => __( 'Parent Organization:', 'ld-organization' ),
				'menu_name'             => __( 'Organizations', 'ld-organization' ),
			),
			'public'                => false,
			'hierarchical'          => false,
			'show_ui'               => true,
			'show_in_nav_menus'     => true,
			'supports'              => array( 'title', 'editor' ),
			'has_archive'           => false,
			'rewrite'               => true,
			'query_var'             => true,
			'menu_position'         => null,
			'menu_icon'             => 'dashicons-groups',
			'show_in_rest'          => true,
			'rest_base'             => 'organization',
			'rest_controller_class' => 'WP_REST_Posts_Controller',
		)
	);

}

add_action( 'init', 'LD_Organization\organization_init' );


/**
 * Handles disabling the editor for the organization post type.
 *
 * @return void
 */
function organization_supports() {
	remove_post_type_support( POST_TYPE_NAME, 'editor' );
}

add_action( 'init', 'LD_Organization\organization_supports' );

/**
 * Sets the post updated messages for the `organization` post type.
 *
 * @param array $messages Post updated messages.
 *
 * @return array Messages for the `organization` post type.
 */
function organization_updated_messages( array $messages ): array {
	global $post;

	$permalink = get_permalink( $post );

	$messages[ POST_TYPE_NAME ] = array(
		0  => '', // Unused. Messages start at index 1.
		/* translators: %s: post permalink */
		1  => sprintf( __( 'Organization updated. <a target="_blank" href="%s">View organization</a>', 'ld-organization' ), esc_url( $permalink ) ),
		2  => __( 'Custom field updated.', 'ld-organization' ),
		3  => __( 'Custom field deleted.', 'ld-organization' ),
		4  => __( 'Organization updated.', 'ld-organization' ),
		/* translators: %s: date and time of the revision */
		5  => isset( $_GET['revision'] ) ? sprintf( __( 'Organization restored to revision from %s', 'ld-organization' ), wp_post_revision_title( (int) $_GET['revision'], false ) ) : false, // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		/* translators: %s: post permalink */
		6  => sprintf( __( 'Organization published. <a href="%s">View organization</a>', 'ld-organization' ), esc_url( $permalink ) ),
		7  => __( 'Organization saved.', 'ld-organization' ),
		/* translators: %s: post permalink */
		8  => sprintf( __( 'Organization submitted. <a target="_blank" href="%s">Preview organization</a>', 'ld-organization' ), esc_url( add_query_arg( 'preview', 'true', $permalink ) ) ),
		/* translators: 1: Publish box date format, see https://secure.php.net/date 2: Post permalink */
		9  => sprintf( __( 'Organization scheduled for: <strong>%1$s</strong>. <a target="_blank" href="%2$s">Preview organization</a>', 'ld-organization' ), date_i18n( __( 'M j, Y @ G:i', 'ld-organization' ), strtotime( $post->post_date ) ), esc_url( $permalink ) ),
		/* translators: %s: post permalink */
		10 => sprintf( __( 'Organization draft updated. <a target="_blank" href="%s">Preview organization</a>', 'ld-organization' ), esc_url( add_query_arg( 'preview', 'true', $permalink ) ) ),
	);

	return $messages;
}

add_filter( 'post_updated_messages', 'LD_Organization\organization_updated_messages' );

/**
 * Sets the bulk post updated messages for the `organization` post type.
 *
 * @param array $bulk_messages Arrays of messages, each keyed by the corresponding post type. Messages are
 *                              keyed with 'updated', 'locked', 'deleted', 'trashed', and 'untrashed'.
 * @param int[] $bulk_counts   Array of item counts for each message, used to build internationalized strings.
 *
 * @return array Bulk messages for the `organization` post type.
 */
function organization_bulk_updated_messages( array $bulk_messages, array $bulk_counts ): array {
	global $post;

	$bulk_messages[ POST_TYPE_NAME ] = array(
		/* translators: %s: Number of organizations. */
		'updated'   => _n( '%s organization updated.', '%s organizations updated.', $bulk_counts['updated'], 'ld-organization' ),
		'locked'    => ( 1 === $bulk_counts['locked'] ) ? __( '1 organization not updated, somebody is editing it.', 'ld-organization' ) :
						/* translators: %s: Number of organizations. */
						_n( '%s organization not updated, somebody is editing it.', '%s organizations not updated, somebody is editing them.', $bulk_counts['locked'], 'ld-organization' ),
		/* translators: %s: Number of organizations. */
		'deleted'   => _n( '%s organization permanently deleted.', '%s organizations permanently deleted.', $bulk_counts['deleted'], 'ld-organization' ),
		/* translators: %s: Number of organizations. */
		'trashed'   => _n( '%s organization moved to the Trash.', '%s organizations moved to the Trash.', $bulk_counts['trashed'], 'ld-organization' ),
		/* translators: %s: Number of organizations. */
		'untrashed' => _n( '%s organization restored from the Trash.', '%s organizations restored from the Trash.', $bulk_counts['untrashed'], 'ld-organization' ),
	);

	return $bulk_messages;
}

add_filter( 'bulk_post_updated_messages', 'LD_Organization\organization_bulk_updated_messages', 10, 2 );
