<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
/**
 * AbstractHook class is an abstract class that provides some sane defaults for registering hooks.
 *
 * @package LD_Organization\Hooks
 */

namespace LD_Organization\Hooks;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use InvalidArgumentException;

/**
 * Handles registering hooks.
 *
 * @since 0.11.0 Added abstract hook class.
 * @package LD_Organization\Hooks
 */
abstract class AbstractHook {

	/**
	 * Registers actions and hooks defined on the object.
	 */
	public function __construct() {
		$this->register_actions();
		$this->register_filters();
	}

	/**
	 * Handles registering the actions.
	 *
	 * @throws InvalidArgumentException If the action hooks are not defined correctly.
	 * @uses AbstractHook::get_actions()
	 * @return void
	 */
	final public function register_actions(): void {
		foreach ( $this->get_actions() as $action ) {
			if ( ! isset( $action['hook'], $action['callable'] ) ) {
				throw new InvalidArgumentException( "Actions are required to have the attributes 'hook' and 'callable' set." );
			}
			add_action( $action['hook'], $action['callable'], $action['priority'] ?? 10, $action['num_args'] ?? 1 );
		}
	}

	/**
	 * Handles registering the filters.
	 *
	 * @throws InvalidArgumentException If the filter hooks are not defined correctly.
	 * @uses AbstractHook::get_filters()
	 * @return void
	 */
	final public function register_filters(): void {
		foreach ( $this->get_filters() as $filter ) {
			if ( ! isset( $filter['hook'], $filter['callable'] ) ) {
				throw new InvalidArgumentException( "Actions are required to have the attributes 'hook' and 'callable' set." );
			}
			add_filter( $filter['hook'], $filter['callable'], $filter['priority'] ?? 10, $filter['num_args'] ?? 1 );
		}
	}

	/**
	 * Override this method and return an array of actions to override.
	 * The array should be in the format of: array( 'hook' => $hook_name, callable' => $callable, 'priority' => $priority, 'num_args' => $num_args )
	 *
	 * @return array
	 * @since 0.11.1 Improved implementation of AbstractHook to use abstract methods.
	 */
	abstract public function get_actions(): array;

	/**
	 * Override this method and return an array of filters to override.
	 * The array should be in the format of: array( 'hook' => $hook_name, callable' => $callable, 'priority' => $priority, 'num_args' => $num_args )
	 *
	 * @return array
	 * @since 0.11.1 Improved implementation of AbstractHook to use abstract methods.
	 */
	abstract public function get_filters(): array;

}
