<?php
/**
 * This file has quiz helper functions
 *
 * @since 0.9.0
 * @package LD_Organization
 */

namespace LD_Organization;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use WpProQuiz_Model_StatisticUser;
use WpProQuiz_Model_StatisticUserMapper;

/**
 * Fetches the statistics based on the ref_id.
 *
 * @param int $ref_id The reference identifier from WP Pro Quiz.
 * @param int $quiz_id The quiz ID used to filter the statistics. (Must be pro_quiz_id).
 *
 * @return array
 * @since 0.9.0
 */
function fetch_quiz_statistics( int $ref_id, int $quiz_id ): array {
	$statistics_mapper = new WpProQuiz_Model_StatisticUserMapper();
	$statistics        = $statistics_mapper->fetchUserStatistic( $ref_id, $quiz_id );

	$final_stats = array();

	foreach ( $statistics as $stat_row ) {
		assert( $stat_row instanceof WpProQuiz_Model_StatisticUser );
		$final_stats[] = array(
			'_correctCount'        => $stat_row->getCorrectCount(),
			'_incorrectCount'      => $stat_row->getIncorrectCount(),
			'_hintCount'           => $stat_row->getHintCount(),
			'_points'              => $stat_row->getPoints(),
			'_questionTime'        => $stat_row->getQuestionTime(),
			'_questionId'          => $stat_row->getQuestionId(),
			'_questionName'        => $stat_row->getQuestionName(),
			'_gPoints'             => $stat_row->getGPoints(),
			'_categoryId'          => $stat_row->getCategoryId(),
			'_categoryName'        => $stat_row->getCategoryName(),
			'_statisticAnswerData' => $stat_row->getStatisticAnswerData(),
			// '_questionAnswerData'  => $stat_row->getQuestionAnswerData(),
			'_answerType'          => $stat_row->getAnswerType(),
		);
	}

	return $final_stats;
}

/**
 * Checks to see if a quiz attempt is passed or not.
 *
 * @see \LDLMS_Model_User_Quiz_Progress::has_completed_step() This function is based on that, but for a single attempt.
 * @param array $quiz_attempt The quiz attempt to check.
 *
 * @return bool
 */
function is_quiz_attempt_passed( array $quiz_attempt ): bool {
	if ( ! isset( $quiz_attempt['quiz'] ) ) {
		return false;
	}

	// Does a check to see if the passed attempt is passed.
	if ( isset( $quiz_attempt['pass'] ) ) {
		$pass = ( 1 === (int) $quiz_attempt['pass'] ) ? 1 : 0;
	} else {
		$quiz_passingpercentage = (int) learndash_get_setting( $quiz_attempt['quiz'], 'passingpercentage' );

		$pass = ( ! empty( $quiz_attempt['count'] ) && $quiz_attempt['score'] * 100 / $quiz_attempt['count'] >= $quiz_passingpercentage ) ? 1 : 0;
	}

	return $pass;
}
