<?php
/**
 * The permissions helper class is a static class that provides permission based methods.
 *
 * @package LD_Organization
 */

namespace LD_Organization;

use InvalidArgumentException;
use WP_REST_Request;

/**
 * Permission helper class for rest requests.
 *
 * @package LD_Organization
 * @since 0.10.0
 */
class RestAPIPermissionHelpers {

	/**
	 * Checks if the user calling the API is an owner of the organization being called.
	 *
	 * @param WP_REST_Request $request The request object for the current request.
	 * @param string          $param_key The parameter key to check.
	 *
	 * @return bool
	 * @uses \LD_Organization\RestAPI::has_access()
	 */
	final public static function is_owner( WP_REST_Request $request, string $param_key = 'id' ): bool {
		return self::has_access( $request, 'organization_owners', $param_key );
	}

	/**
	 * Checks if the user calling the API is an owner of the organization being called.
	 *
	 * @param WP_REST_Request $request The request object for the current request.
	 * @param string          $param_key The parameter key to check for organization.
	 *
	 * @return bool
	 * @uses \LD_Organization\RestAPI::has_access()
	 */
	final public static function is_group_leader( WP_REST_Request $request, string $param_key = 'id' ): bool {
		return self::has_access( $request, 'group_leaders', $param_key );
	}

	/**
	 * Handles checking if the user is logged in and group leader for the requested course.
	 *
	 * @param WP_REST_Request $request The current WP_REST_Request object.
	 *
	 * @return bool
	 */
	final public static function is_group_leader_of_post( WP_REST_Request $request ): bool {
		if ( ! is_user_logged_in() ) {
			return false;
		}

		$current_user = wp_get_current_user();

		if ( null === $current_user ) {
			return false;
		}

		// Current user is an admin user, allow all.
		if ( current_user_can( 'manage_options' ) ) {
			return true;
		}

		// If we pass the WP checks, we check if the user is a group leader and is a part of the group administrators.
		$admins = learndash_get_groups_administrator_ids( $request->get_param( 'group_id' ) );

		if ( ! in_array( $current_user->ID, $admins, true ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Checks if the user is logged in.
	 *
	 * @return bool
	 */
	final public static function is_logged_in(): bool {
		return is_user_logged_in();
	}

	/**
	 * Handles checking if the organization is owned by the authenticated user.
	 *
	 * @param WP_REST_Request $request The request with a required parameter of ID.
	 * @param string          $role The role to check against.
	 * @param string          $param_key Optionally the key of the parameter to check.
	 *
	 * @return bool
	 * @uses \LD_Organization\Organization::with_id(),\LD_Organization\Organization::has_user()
	 */
	final public static function has_access( WP_REST_Request $request, string $role = 'both', string $param_key = 'id' ): bool {
		if ( ! is_user_logged_in() ) {
			return false;
		}
		$id = $request->get_param( $param_key );
		if ( ! isset( $id ) ) {
			return false;
		}
		try {
			$organization = Organization::with_id( $id );
			$current_user = wp_get_current_user();
			if ( isset( $current_user ) && $organization->has_user( $current_user->ID, $role ) ) {
				return true;
			}
		} catch ( OrganizationNotFoundException | InvalidArgumentException $exception ) {
			return false;
		}

		return false;
	}
}
