<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase

/**
 * Handles registering the task page for performing certain helper tasks.
 *
 * @package LD_Organization\Admin
 */

namespace LD_Organization\Admin;

/**
 * Registers the Task page.
 *
 * @package LD_Organization\Admin
 */
class HelperTasks extends AbstractAdminPage {

	public int $menu_priority = 99;

	/**
	 * Handles adding the management page.
	 *
	 * @return void
	 */
	final public function register_pages(): void {
		add_submenu_page(
			'ld-organization-settings',
			'Helper Tasks',
			'Helper Tasks',
			'manage_options',
			'ld-helper-tasks',
			array( $this, 'page_callback' )
		);
	}

	/**
	 * Handles outputting the html.
	 *
	 * @return void
	 */
	final public function page_callback(): void {
		?>

		<div class="wrap">
			<h2>LD-Organization - Helper Tasks</h2>
			<p>Different tools and utilities to handle minor tasks.</p>
			<?php settings_errors(); ?>

			<div class="flex flex-col gap-2 mt-4" x-data="data()">
				<script>
					function data() {
						return {
							loading: {
								recalculate: false,
								unlinked: false
							},
							unlinked_groups: [],
							handle_fetch(endpoint, config = {method: 'GET'}) {
								return fetch(`${admin_extras.baseAPI}${endpoint}`,
									{
										...config,
										headers: {
											'X-WP-Nonce': admin_extras.nonce
										}
									}
								).then((r) => {
									if (!r.ok) {
										return Promise.reject(r);
									}
									return Promise.resolve(r);
								}).then(r => r.json());
							}
						}
					}
				</script>
				<div class="flex gap-x-3 w-full">
					<button type="button"
							class="flex items-center rounded-md text-white bg-blue-500 px-4 py-2 hover:bg-blue-600 transition-colors duration-100 "
							@click="
					loading.unlinked = true;
					handle_fetch('/admin/unlinked_groups')
						.then(r => {
								if (r.success) {
									toastr.success(r.message);
								}
								loading.unlinked = false;
								if (r.data) {
									unlinked_groups = r.data;
								}
						})
						.catch(() => {
							toastr.error('An error occured')
							loading.unlinked = false;
						})
					">
						<svg x-show="loading.unlinked" xmlns="http://www.w3.org/2000/svg"
							 class="animate-spin mr-2 h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"
							 stroke-width="2">
							<path stroke-linecap="round" stroke-linejoin="round"
								  d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/>
						</svg>
						List all unlinked groups
					</button>
					<button type="button"
							class="flex items-center rounded-md text-white bg-blue-500 px-4 py-2 hover:bg-blue-600 transition-colors duration-100 "
							@click="
					loading.recalculate = true;
					handle_fetch('/admin/recalculate_licenses', {method: 'POST'})
						.then(r => {
								if (r.success) {
									toastr.success(r.message);
								}
								loading.recalculate = false;
						})
						.catch(() => {
							toastr.error('An error occured')
							loading.recalculate = false;
						})
					">
						<svg x-show="loading.recalculate" xmlns="http://www.w3.org/2000/svg"
							 class="animate-spin mr-2 h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"
							 stroke-width="2">
							<path stroke-linecap="round" stroke-linejoin="round"
								  d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/>
						</svg>
						Recalculate all licenses
					</button>
				</div>
				<div class="bg-gray flex" x-show="unlinked_groups.length > 0">
					<ul>
						<template x-for="post in unlinked_groups">
							<li class="flex justify-between border border-sky-500 py-3 px-1">
								<span x-text="post.post_title"></span>
								<a class="text-underline text-blue-500"
								   x-bind:href="`/wp-admin/post.php?post=${post.ID}&action=edit`" target="_blank">
									Edit
								</a>
							</li>
						</template>
					</ul>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Handles checking if the post is an upload.
	 *
	 * @return void
	 */
	final public function handle_post(): void {
	}

	/**
	 * Handles loading required scripts.
	 *
	 * @param string $hook the current page.
	 *
	 * @return void
	 */
	final public function enqueue_scripts( string $hook ): void {
		if ( 'ld-organization-settings_page_ld-helper-tasks' !== $hook ) {
			return;
		}

		// Tailwind.
		wp_enqueue_script( 'tailwind-play', 'https://cdn.tailwindcss.com', array(), '1', false );

		// Toaster.
		wp_enqueue_script( 'toastr-js', 'https://cdnjs.cloudflare.com/ajax/libs/toastr.js/latest/toastr.min.js', array(), '1', false );
		wp_enqueue_style( 'toastr-css', 'https://cdnjs.cloudflare.com/ajax/libs/toastr.js/latest/toastr.min.css', array(), '1' );

		// Alpine.
		wp_enqueue_script( 'alpine-js', 'https://unpkg.com/alpinejs', array(), '1', true );
		wp_localize_script(
			'alpine-js',
			'admin_extras',
			array(
				'nonce'   => wp_create_nonce( 'wp_rest' ),
				'baseAPI' => get_rest_url( null, '/ld-organization/v1' ),
			)
		);
	}

}
