<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
// phpcs:disable Generic.Commenting.DocComment.MissingShort

namespace LD_Organization\Hooks;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use DateTime;
use InvalidArgumentException;
use LD_Organization\Exceptions\OrganizationNotFoundException;
use LD_Organization\Organization;
use LDLMS_DB;
use function Sentry\captureException;

/**
 * Handles calling the Organization for the current group and updating the Organizations licenses.
 *
 * @package LD_Organization\Hooks
 * @since 0.12.0 Added Class
 */
class OrganizationRunnerHook extends AbstractHook {

	/**
	 * Resets all completions.
	 * Also stores the deleted completions in a meta field.
	 *
	 * @return string[]
	 */
	final public static function reset_completions(): array {
		global $wpdb;
		$meta_ids = $wpdb->get_col( $wpdb->prepare( "SELECT meta_id FROM {$wpdb->postmeta} WHERE meta_key = %s;", Organization::ORGANIZATION_COMPLETIONS_META ) );
		foreach ( $meta_ids as $meta_id ) {
			$old_data = get_metadata_by_mid( 'post', $meta_id );
			if ( ! empty( $old_data ) ) {
				printf( "archiving %s\n", esc_html( $old_data->post_id ) );
				update_post_meta( $old_data->post_id, Organization::ORGANIZATION_COMPLETIONS_META . '_archived', $old_data->meta_value ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			}
		}
		delete_post_meta_by_key( Organization::ORGANIZATION_COMPLETIONS_META );

		return array(
			'status' => 'ok',
		);
	}

	/**
	 * Handles restoring old archived completions.
	 *
	 * @return string[]
	 */
	final public static function restore_archived_completions(): array {
		global $wpdb;
		$meta_ids = $wpdb->get_col( $wpdb->prepare( "SELECT meta_id FROM {$wpdb->postmeta} WHERE meta_key = %s;", Organization::ORGANIZATION_COMPLETIONS_META . '_archived' ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		foreach ( $meta_ids as $meta_id ) {
			$old_data = get_metadata_by_mid( 'post', $meta_id );
			if ( ! empty( $old_data ) ) {
				printf( "restoring %s\n", esc_html( $old_data->post_id ) );
				$result = update_post_meta( $old_data->post_id, Organization::ORGANIZATION_COMPLETIONS_META, $old_data->meta_value );
			}
			if ( ! isset( $result ) || ! $result ) {
				printf( "failed to restore %s\n", esc_html( $old_data->post_id ) );
				continue;
			}
			delete_post_meta( $old_data->post_id, Organization::ORGANIZATION_COMPLETIONS_META . '_archived' );
		}

		return array(
			'status' => 'ok',
		);
	}

	/**
	 * Handles checking if the group belongs to any organizations, and if so, updates the license information.
	 *
	 * @param array $args The arguments passed to the function.
	 *
	 * @return array
	 */
	final public static function add_completions( $args ): array {
		global $wpdb;
		if ( ! class_exists( '\LDLMS_DB' ) ) {
			return array(
				'status'  => 'error',
				'message' => 'Learndash not installed',
			);
		}
		$table_name = LDLMS_DB::get_table_name( 'user_activity' );
		$page       = $args['page'] ?? 1;
		$page       = absint( $page );
		$offset     = ( $page - 1 ) * 100;
		$data       = $wpdb->get_results( "SELECT * FROM {$table_name} WHERE activity_type = 'course' AND activity_completed != 0 LIMIT 100 OFFSET {$offset};", ARRAY_A );
		printf( "parsing page %s\n", $page );
		$user_cache = array();
		foreach ( $data as $row ) {
			$course_id = $row['course_id'];
			$user_id   = $row['user_id'];
			$groups    = learndash_get_users_group_ids( $user_id );
			if ( empty( $user_cache[ $user_id ] ) ) {
				$organizations = array();
				foreach ( $groups as $group ) {
					try {
						$organization    = Organization::by_group_id( $group );
						$organizations[] = $organization;
					} catch ( OrganizationNotFoundException | InvalidArgumentException $e ) {
						continue;
					}
				}
			}
			if ( empty( $organizations ) ) {
				continue;
			}
			$user_cache[ $user_id ] = $organizations;
			$year                   = ( new DateTime( "@{$row['activity_completed']}" ) )->format( 'Y' );
			foreach ( $organizations as $organization ) {
				try {
					$organization->add_completion(
						$year,
						array(
							'course_id' => $course_id,
							'user_id'   => $user_id,
						)
					);
				} catch ( OrganizationNotFoundException $e ) {
					captureException( $e );
				}
			}
		}
		if ( empty( $data ) ) {
			return array(
				'status' => 'ok',
			);
		}

		return array(
			'status'    => 'ok',
			'next_page' => $page + 1,
		);
	}

	/**
	 * Adds the runner function to the list of functions.
	 *
	 * @param array $functions The current list of functions.
	 *
	 * @return array
	 */
	final public function add_runner_function( array $functions ): array {
		$functions['add_completions']     = array(
			'title'    => 'Add old completions',
			'callback' => '\\' . __CLASS__ . '::add_completions',
		);
		$functions['reset_completions']   = array(
			'title'    => 'Remove all completions',
			'callback' => '\\' . __CLASS__ . '::reset_completions',
		);
		$functions['restore_completions'] = array(
			'title'    => 'Restore all completions',
			'callback' => '\\' . __CLASS__ . '::restore_archived_completions',
		);

		return $functions;
	}

	/**
	 * @inheritDoc
	 */
	public function get_actions(): array {
		return array();
	}

	/**
	 * @inheritDoc
	 */
	public function get_filters(): array {
		return array(
			array(
				'hook'     => 'jcore_runner_functions',
				'callable' => array( $this, 'add_runner_function' ),
				'priority' => 10,
				'num_args' => 1,
			),
		);
	}
}
