<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
// phpcs:disable Generic.Commenting.DocComment.MissingShort

namespace LD_Organization\Hooks;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use DateTime;
use InvalidArgumentException;
use LD_Organization\Exceptions\OrganizationNotFoundException;
use LD_Organization\Organization;
use WP_Post;
use WP_User;
use function Sentry\captureException;

/**
 * Handles calling the Organization for the current group and updating the Organizations licenses.
 *
 * @package LD_Organization\Hooks
 * @since 0.12.0 Added Class
 */
class OrganizationCompletions extends AbstractHook {

	/**
	 * Handles checking if the group belongs to any organizations, and if so, updates the license information.
	 *
	 * @param array $args {
	 *     Arguments passed from learndash.
	 *
	 * @type false|null|WP_User $user The user that completed the course.
	 * @type array|null|WP_Post $course The course that was completed.
	 * @type array $progress The progress array.
	 * @type int $course_completed The timestamp when the course was completed.
	 * }
	 *
	 * @return void
	 */
	final public function add_organization_completion( array $args ): void {
		$user   = $args['user'];
		$course = $args['course'];
		if ( ! $user instanceof WP_User || ! $course instanceof WP_Post ) {
			return;
		}
		// Get the groups the user is in.
		$groups = learndash_get_users_group_ids( $user->ID, true );
		// Get the organizations the user is in.
		$organizations = array();
		foreach ( $groups as $group ) {
			try {
				$organization    = Organization::by_group_id( $group );
				$organizations[] = $organization;
			} catch ( OrganizationNotFoundException|InvalidArgumentException $e ) {
				continue;
			}
		}

		if ( empty( $organizations ) ) {
			return;
		}

		$year = ( new DateTime() )->format( 'Y' );

		foreach ( $organizations as $organization ) {
			try {
				$organization->add_completion(
					$year,
					array(
						'course_id' => $course->ID,
						'user_id'   => $user->ID,
					)
				);
			} catch ( OrganizationNotFoundException $e ) {
				captureException( $e );
			}
		}
	}


	/**
	 * @inheritDoc
	 */
	public function get_actions(): array {
		return array(
			array(
				'hook'     => 'learndash_course_completed',
				'callable' => array( $this, 'add_organization_completion' ),
				'priority' => 10,
				'num_args' => 1,
			),
		);
	}

	/**
	 * @inheritDoc
	 */
	public function get_filters(): array {
		return array();
	}
}
