<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
/**
 * Handles validation for rest API requests.
 *
 * @package LD_Organization
 * @since 0.10.0
 */

namespace LD_Organization;

/**
 * A helper class for validation of rest API requests.
 *
 * @package LD_Organization
 * @since 0.10.0 Added external static class for handling rest API requests validation arguments..
 */
class RestAPIValidationHelpers {

	/**
	 * The default validate_callback if the endpoint has an ID in it.
	 *
	 * @param array $args An array of arguments to override/add to the defaults.
	 *
	 * @return array
	 * @since 0.10.0 Added $args to allow override, also changed return structure.
	 */
	final public static function validate_id_argument( array $args = array() ): array {
		$defaults = array(
			'validate_callback' => function ( $param, $request, $key ) {
				return is_numeric( $param );
			},
		);

		return wp_parse_args( $args, $defaults );
	}

	/**
	 * Checks if the parameter is a valid group_id.
	 *
	 * @param array $args Any arguments to override the defaults.
	 *
	 * @return array
	 * @since 0.10.0 Added $args parameter.
	 */
	final public static function is_valid_ld_group_id( array $args = array() ): array {
		$defaults = array(
			'validate_callback' => function ( $param, $request, $key ) {
				return is_numeric( $param ) && learndash_is_group_post( $param );
			},
		);

		return wp_parse_args( $args, $defaults );
	}

	/**
	 * Checks if the parameter is a valid User ID.
	 *
	 * @param array $args An array of arguments to override/add to the defaults.
	 *
	 * @return array
	 * @since 0.10.0 Added $args parameter.
	 */
	final public static function is_valid_user_id( array $args = array() ): array {
		$defaults = array(
			'validate_callback' => function ( $param, $request, $key ) {
				return is_numeric( $param ) && false !== get_user_by( 'ID', $param );
			},
		);

		return wp_parse_args( $args, $defaults );
	}

	/**
	 * Checks if the parameter is a valid group post ID.
	 *
	 * @param array $args An array of arguments to override/add to the defaults.
	 *
	 * @return array
	 * @since 0.10.0 Added $args parameter.
	 */
	final public static function is_valid_ld_course_id( array $args = array() ): array {
		$defaults = array(
			'validate_callback' => function ( $param, $request, $key ) {
				return is_numeric( $param ) && learndash_is_course_post( $param );
			},
		);

		return wp_parse_args( $args, $defaults );
	}

	/**
	 * Checks if the parameter is a valid quiz post ID.
	 *
	 * @param array $args An array of arguments to override/add to the defaults.
	 *
	 * @return array
	 * @since 0.10.0 Added $args parameter.
	 */
	final public static function is_valid_ld_quiz_id( array $args = array() ): array {
		$defaults = array(
			'validate_callback' => function ( $param, $request, $key ) {
				return is_numeric( $param ) && learndash_get_post_type_slug( 'quiz' ) === get_post_type( $param );
			},
		);

		return wp_parse_args( $args, $defaults );
	}

	/**
	 * Checks if the parameter is a valid organization ID.
	 *
	 * @param array $args An array of arguments to override/add to the defaults.
	 *
	 * @return array
	 * @since 0.10.0 Added $args parameter.
	 */
	final public static function is_valid_org_id( array $args = array() ): array {
		$defaults = array(
			'validate_callback' => function ( $param, $request, $key ) {
				return is_numeric( $param ) && is_valid_organization_post( $param );
			},
		);

		return wp_parse_args( $args, $defaults );
	}
}
