<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
/**
 * Transients functionality for caching expensive operations.
 *
 * @package LD_Organization
 * @since 0.5.0
 */

namespace LD_Organization;

/**
 * A static class to handle transients, has some sane defaults.
 *
 * @package LD_Organization
 * @since 0.5.0
 */
class LDOrgTransientManager {
	/**
	 * The namespace for the transient.
	 *
	 * @var string
	 */
	public static string $ns = 'ld_org';

	/**
	 * The default ttl for the transients. (10 minutes in seconds).
	 *
	 * @var int
	 */
	public static int $default_ttl = 10 * MINUTE_IN_SECONDS;

	/**
	 * Gets the transient, transforms the name to a properly namespace-qualified name.
	 *
	 * @param string $name The name of the transient to fetch for.
	 *
	 * @return mixed
	 */
	public static function get( string $name ) {
		return get_transient( self::build_name( $name ) );
	}

	/**
	 * Sets the transient, transforms the name to a properly namespace-qualified name.
	 *
	 * @param string   $name The name of the transient to store.
	 * @param mixed    $value The value of the transient.
	 * @param null|int $ttl Optionally an override to the default ttl.
	 *
	 * @return bool
	 */
	public static function set( string $name, $value, int $ttl = null ) {
		if ( null === $ttl ) {
			$ttl = (int) self::$default_ttl;
		}

		return set_transient( self::build_name( $name ), $value, $ttl );
	}

	/**
	 * Deletes the transient for the given name.
	 *
	 * @param string $name The name of the transient to delete.
	 *
	 * @return bool
	 */
	public static function delete( string $name ): bool {
		return delete_transient( self::build_name( $name ) );
	}

	/**
	 * Handles building the transient string out of the given name and the namespace. eg "ld_org_$name".
	 *
	 * @param string $name the transient name.
	 *
	 * @return string
	 */
	private static function build_name( string $name ): string {
		return self::$ns . '_' . $name;
	}
}
