<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase

namespace LD_Organization\Hooks;

use DateTime;
use DateTimeZone;
use Exception;
use LD_Organization\Data\EssayExportProcessingData;
use Monolog\Logger;
use WP_Post;
use const LD_Organization\LD_ORG_PROCESSING_TRANSIENT_KEY;

/**
 * Class ExportEssayAjax
 *
 * @package LD_Organization\Hooks
 * @since 0.20.0
 */
class ExportEssayAjax extends AbstractHook {
	/**
	 * Per page.
	 *
	 * @var int
	 */
	public const PER_PAGE = 50;

	/**
	 * Logger instance
	 *
	 * @var Logger
	 */
	private Logger $logger;

	/**
	 * Per request
	 *
	 * @var int
	 */
	private int $per_request = self::PER_PAGE;
	/**
	 * The transient key for the processing ids.
	 *
	 * @var string
	 */
	private string $transient_key = LD_ORG_PROCESSING_TRANSIENT_KEY;
	/**
	 * Prefix for the processing data transient key.
	 *
	 * @var string
	 */
	private string $data_prefix = '_processing_data_';

	/**
	 * Sets up the class
	 */
	public function __construct() {
		parent::__construct();
		$this->logger = logger( 'ExportEssayAjax' );
	}

	/**
	 * @inheritDoc
	 */
	final public function get_actions(): array {
		return array(
			array(
				'hook'     => 'wp_ajax_export_essay',
				'callable' => array( $this, 'handle_ajax_export' ),
				'priority' => 10,
				'num_args' => 0,
			),
			array(
				'hook'     => 'wp_ajax_fetch_exports',
				'callable' => array( $this, 'handle_ajax_fetch_exports' ),
				'priority' => 10,
				'num_args' => 0,
			),
		);
	}

	/**
	 * @inheritDoc
	 */
	final public function get_filters(): array {
		return array();
	}

	/**
	 * Handles the export_essay ajax request.
	 *
	 * @return void
	 */
	final public function handle_ajax_export(): void {
		if ( ! learndash_is_group_leader_user() && ! learndash_is_admin_user() ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_permission',
				)
			);
		}
		$nonce = sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'ld-essay-nonce' ) ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_nonce',
				)
			);
		}
		// Gets the current processing id.
		$processing_id = sanitize_text_field( wp_unslash( $_POST['processing_id'] ?? '' ) );

		// Checks if the user wants to download the export (is it done?).
		$download = sanitize_text_field( wp_unslash( $_POST['download'] ?? '' ) );
		if ( ! empty( $download ) && 'yes' === $download ) {
			$this->download_essay( $processing_id );

			return;
		}

		if ( ! empty( $processing_id ) ) {
			$this->process_export_essay( $processing_id );

			return;
		}

		$start_date = sanitize_text_field( wp_unslash( $_POST['start_date'] ?? '' ) );
		$end_date   = sanitize_text_field( wp_unslash( $_POST['end_date'] ?? '' ) );
		$dates      = array();
		if ( ! empty( $start_date ) && ! empty( $end_date ) ) {
			$dates = $this->handle_dates( $start_date, $end_date );
		}
		if ( ! empty( $dates ) && count( $dates ) === 2 ) {
			[ $start_date, $end_date ] = $dates;
		}
		$this->start_export_essay( $start_date, $end_date );
	}

	/**
	 * Handles downloading the export_essay ajax request.
	 *
	 * @param string $processing_id The processing id.
	 *
	 * @return void
	 */
	private function download_essay( string $processing_id ): void {
		$processing_data = $this->get_processing_data( $processing_id );
		if ( ! $processing_data ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_processing_id',
				),
				400
			);
		}
		if ( $processing_data->user_id !== get_current_user_id() ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_permission',
				),
				400
			);
		}
		if ( ! $processing_data->is_done() ) {
			wp_send_json_error(
				array(
					'error' => 'export_not_done',
				),
				400
			);
		}
		$file = $processing_data->get_file();
		if ( ! file_exists( $file ) ) {
			wp_send_json_error(
				array(
					'error' => 'file_not_found',
				),
				404
			);
		}
		$fs = fopen( $file, 'rb' );
		if ( ! $fs ) {
			wp_send_json_error(
				array(
					'error' => 'file_not_found',
				),
				404
			);
		}
		header( 'Content-Type: text/csv' );
		header( 'Content-Disposition: attachment; filename=' . basename( $file ) );
		fpassthru( $fs );
		fclose( $fs );
		wp_die();
	}

	/**
	 * Returns the processing data for the given processing id.
	 *
	 * @param string $processing_id The processing id.
	 *
	 * @return EssayExportProcessingData|false
	 */
	private function get_processing_data( string $processing_id ): EssayExportProcessingData|false {
		$processing_ids = get_transient( $this->transient_key );
		if ( empty( $processing_ids ) || ! is_array( $processing_ids ) ) {
			return false;
		}
		if ( ! array_key_exists( $processing_id, $processing_ids ) ) {
			return false;
		}
		$processing_data = get_transient( $this->data_prefix . $processing_id );
		if ( empty( $processing_data ) || ! is_a( $processing_data, EssayExportProcessingData::class ) ) {
			return false;
		}

		return $processing_data;
	}

	/**
	 * Continues the export_essay ajax request.
	 *
	 * @param string $processing_id The processing id.
	 *
	 * @return void
	 */
	private function process_export_essay( string $processing_id ): void {
		$processing_data = $this->get_processing_data( $processing_id );
		if ( ! $processing_data ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_processing_id',
				)
			);
		}
		if ( $processing_data->is_done() ) {
			wp_send_json_success(
				array(
					'processing_id' => $processing_id,
					'continue'      => false,
					'done'          => true,
					'empty'         => $processing_data->empty,
				)
			);
		}
		$export_data = $this->export_essay( $processing_data );
		if ( ! $export_data ) {
			wp_send_json_error(
				array(
					'error' => 'export_failed',
				)
			);
		}
		wp_send_json_success(
			array(
				'processing_id' => $processing_id,
				'continue'      => ! $export_data->is_done(),
				'done'          => $export_data->is_done(),
				'empty'         => $export_data->empty,
				'count'         => $export_data->total_posts,
				'per_page'      => $export_data->per_page,
			)
		);
	}

	/**
	 * Handles exporting the essay.
	 *
	 * @param EssayExportProcessingData $processing_data The processing data.
	 *
	 * @return EssayExportProcessingData|false
	 */
	private function export_essay( EssayExportProcessingData $processing_data ): EssayExportProcessingData|false {
		$course_ids      = $processing_data->course_ids;
		$user_ids        = $processing_data->user_ids;
		$users_group_ids = $processing_data->group_ids;
		if ( empty( $user_ids ) && empty( $users_group_ids ) ) {
			$users_group_ids = learndash_get_administrators_group_ids( $processing_data->user_id );
		}
		if ( empty( $user_ids ) && empty( $users_group_ids ) ) {
			return false;
		}
		if ( empty( $user_ids ) ) {
			$user_ids = array_values(
				array_map(
					static function ( $group_id ) {
						return learndash_get_groups_user_ids( $group_id );
					},
					$users_group_ids
				)
			);
			if ( ! empty( $user_ids ) ) {
				$user_ids = array_unique( array_merge( ...$user_ids ) );
			}
		}
		if ( empty( $course_ids ) ) {
			$course_ids = array_values(
				array_map(
					static function ( $group_id ) {
						return learndash_group_enrolled_courses( $group_id, true );
					},
					$users_group_ids
				)
			);
			if ( ! empty( $course_ids ) ) {
				$course_ids = array_unique( array_merge( ...$course_ids ) );
			}
		}
		if ( empty( $course_ids ) || empty( $user_ids ) ) {
			$this->logger->error(
				'No users or courses found.',
				array(
					'requesting_user' => $processing_data->user_id,
					'user_ids'        => $user_ids,
					'course_ids'      => $course_ids,
				)
			);

			return false;
		}
		$processing_data->user_ids   = $user_ids;
		$processing_data->course_ids = $course_ids;
		$processing_data->group_ids  = $users_group_ids;

		$start_date = $processing_data->start_date;
		$end_date   = $processing_data->end_date;

		$essay_type = learndash_get_post_type_slug( 'essay' );
		$statuses   = array( 'graded', 'not_graded' );
		$args       = array(
			'post_type'   => $essay_type,
			'post_status' => $statuses,
			'orderby'     => 'ID',
			'order'       => 'ASC',
			'numberposts' => $this->per_request,
			'offset'      => $processing_data->get_iteration(),
			'author__in'  => $user_ids,
			'meta_query'  => array( // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
				array(
					'key'     => 'course_id',
					'value'   => $course_ids,
					'compare' => 'IN',
				),
			),
		);

		if ( isset( $start_date, $end_date ) ) {
			$args['date_query'] = array(
				array(
					'after'  => $start_date->format( 'Y-m-d' ),
					'before' => $end_date->format( 'Y-m-d' ),
				),
			);
		}
		$this->logger->info( 'Fetching posts.', array( 'args' => $args ) );
		$posts                        = get_posts( $args );
		$processing_data->total_posts += count( $posts );
		if ( empty( $posts ) && $processing_data->get_iteration() === 0 ) {
			$this->logger->info( 'No posts found.', array( 'args' => $args ) );
			$processing_data->set_done( true );
			$processing_data->save();

			return $processing_data;
		}

		$progress = $this->write_csv( $processing_data, $posts );
		if ( ! $progress ) {
			$this->logger->error( 'Failed to write CSV file.', array( 'processing_id' => $processing_data->processing_id ) );

			return false;
		}

		if ( count( $posts ) === $this->per_request ) {
			$this->logger->debug(
				'Incrementing iteration',
				array(
					'processing_id' => $processing_data->processing_id,
					'iteration'     => $processing_data->get_iteration(),
				)
			);
			$processing_data->increment_iteration();
		} else {
			$this->logger->debug(
				'Finalizing export',
				array(
					'processing_id' => $processing_data->processing_id,
					'iteration'     => $processing_data->get_iteration(),
				)
			);
			$processing_data->set_done();
		}

		$processing_data->save();

		return $processing_data;
	}

	/**
	 * Writes the CSV file.
	 *
	 * @param EssayExportProcessingData $processing_data The processing data.
	 * @param array                     $posts The posts.
	 *
	 * @return bool
	 */
	private function write_csv( EssayExportProcessingData $processing_data, array $posts ): bool {
		if ( empty( $posts ) ) {
			return true;
		}
		$file_path = $processing_data->get_file();
		if ( ! $file_path ) {
			$this->logger->error( 'Failed to get file path.', array( 'processing_id' => $processing_data->processing_id ) );

			return false;
		}
		if ( $processing_data->get_iteration() !== 0 && ! file_exists( $file_path ) ) {
			return false;
		}
		$headers = array( 'id', 'title', 'content', 'author', 'submitted_at', 'groups', 'course', 'quiz', 'question' );
		if ( $processing_data->get_iteration() === 0 ) {
			$file = fopen( $file_path, 'wb' );
			if ( ! $file ) {
				$this->logger->error( 'Failed to open file.', array( 'file_path' => $file_path ) );

				return false;
			}
			$success = fputcsv( $file, $headers );
			fclose( $file );
			if ( ! $success ) {
				$this->logger->error( 'Failed to write headers.', array( 'file_path' => $file_path ) );

				return false;
			}
		}
		$file = fopen( $file_path, 'ab' );
		foreach ( $posts as $post ) {
			if ( ! $post instanceof WP_Post ) {
				$this->logger->error(
					'Post is not a WP_Post object.',
					array(
						'processing_id' => $processing_data->processing_id,
						'post'          => $post,
					)
				);
				continue;
			}
			$processed_data = $this->process_post( $post );
			if ( count( $processed_data ) !== count( $headers ) ) {
				$this->logger->error(
					'Processed data does not match headers.',
					array(
						'processing_id'  => $processing_data->processing_id,
						'post'           => $post,
						'headers'        => $headers,
						'processed_data' => $processed_data,
					)
				);
				continue;
			}
			$progress = fputcsv( $file, $processed_data );
			if ( ! $progress ) {
				$this->logger->error( 'Failed to write to file.', array( 'file_path' => $file_path ) );
				break;
			}
		}
		if ( ! $progress ) {
			return false;
		}

		return true;
	}

	/**
	 * Handles processing a post. (Post ID, Title, Content, Author, Submitted At, Course, Quiz, Question)
	 *
	 * @param WP_Post $post The post to process.
	 *
	 * @return array
	 */
	private function process_post( WP_Post $post ): array {
		$processed_data   = array();
		$processed_data[] = $post->ID; // Post ID.
		$processed_data[] = $post->post_title; // Title.
		$processed_data[] = $post->post_content; // Content.
		// Author.
		$author = $post->post_author;
		$user   = get_user_by( 'id', $author );
		if ( $user ) {
			$processed_data[] = get_user_meta( $user->ID, 'first_name', true ) . ' ' . get_user_meta( $user->ID, 'last_name', true );
		} else {
			$processed_data[] = $post->post_author;
		}
		// Submitted at.
		try {
			$date             = new DateTime( $post->post_date, new DateTimeZone( 'Europe/Helsinki' ) );
			$processed_data[] = $date->format( 'd.m.Y H:i:s' );
		} catch ( Exception $e ) {
			$processed_data[] = '';
		}
		// Groups.
		if ( $user ) {
			$groups = learndash_get_users_group_ids( $user->ID, true );
			if ( ! empty( $groups ) ) {
				$group_names      = array_values(
					array_map(
						static function ( $group_id ) {
							$group = get_post( $group_id );

							return $group->post_title ?? '';
						},
						$groups
					)
				);
				$processed_data[] = implode( ', ', $group_names );
			} else {
				$processed_data[] = '';
			}
		} else {
			$processed_data[] = '';
		}
		// Course.
		$course_id = get_post_meta( $post->ID, 'course_id', true );
		if ( $course_id ) {
			$course = get_post( $course_id );
			if ( $course ) {
				$processed_data[] = $course->post_title;
			} else {
				$processed_data[] = '';
			}
		} else {
			$processed_data[] = '';
		}
		// Quiz.
		$quiz_id = get_post_meta( $post->ID, 'quiz_post_id', true );
		if ( $quiz_id ) {
			$quiz = get_post( $quiz_id );
			if ( $quiz ) {
				$processed_data[] = $quiz->post_title;
			} else {
				$processed_data[] = '';
			}
		} else {
			$processed_data[] = '';
		}
		// Question.
		$question_id = get_post_meta( $post->ID, 'question_post_id', true );
		if ( $question_id ) {
			$question = get_post( $question_id );
			if ( $question ) {
				$processed_data[] = $question->post_title;
			} else {
				$processed_data[] = '';
			}
		} else {
			$processed_data[] = '';
		}

		return $processed_data;
	}

	/**
	 * Handles parsing the dates from the request.
	 *
	 * @param string $start_date_str The start date.
	 * @param string $end_date_str The end date.
	 *
	 * @return array
	 */
	private function handle_dates( string $start_date_str, string $end_date_str ): array {
		try {
			$start_date = new DateTime( $start_date_str );
			$end_date   = new DateTime( $end_date_str );
			$start_date->setTime( 0, 0 );
			$end_date->setTime( 23, 59, 59 );
		} catch ( Exception $e ) {
			$this->logger->error( 'Failed to parse dates.', array( 'error' => $e->getMessage() ) );
			return array();
		}

		return array( $start_date, $end_date );
	}

	/**
	 * Starts the export_essay ajax request.
	 *
	 * @param string|DateTime $start_date The start date.
	 * @param string|DateTime $end_date The end date.
	 *
	 * @return void
	 */
	private function start_export_essay( string|DateTime $start_date, string|DateTime $end_date ): void {
		$processing_id = $this->create_processing_id();

		$final_start_date = is_a( $start_date, DateTime::class ) ? $start_date : null;
		$final_end_date   = is_a( $end_date, DateTime::class ) ? $end_date : null;

		$initial_data = new EssayExportProcessingData(
			$processing_id,
			get_current_user_id(),
			$final_start_date,
			$final_end_date,
		);

		$export_data = $this->export_essay( $initial_data );

		if ( ! $export_data ) {
			wp_send_json_error(
				array(
					'error' => 'export_failed',
				)
			);
		}

		wp_send_json_success(
			array(
				'processing_id' => $processing_id,
				'continue'      => ! $export_data->is_done(),
				'done'          => $export_data->is_done(),
				'empty'         => $export_data->empty,
				'count'         => $export_data->total_posts,
				'per_page'      => $export_data->per_page,
			)
		);
	}

	/**
	 * Handles generating a new processing id.
	 *
	 * @return string
	 */
	private function create_processing_id(): string {
		$user_id = get_current_user_id();
		$new_id  = uniqid( '', true );
		$exists  = get_transient( $this->transient_key );
		if ( ! is_array( $exists ) ) {
			$exists = array();
		}
		if ( empty( $exists ) ) {
			// The transient is empty (or expired), so we can create a new id.
			$exists = array(
				$new_id => $user_id,
			);
			set_transient( $this->transient_key, $exists, DAY_IN_SECONDS );

			return $new_id;
		}
		// The transient is not empty, so we need to check if the new id already exists.
		if ( array_key_exists( $new_id, $exists ) ) {
			// The new id already exists, so we go again.
			return $this->create_processing_id();
		}
		// The new id does not exist, so we add it to the transient.
		$exists[ $new_id ] = $user_id;
		set_transient( $this->transient_key, $exists, DAY_IN_SECONDS );

		return $new_id;
	}

	/**
	 * Handles fetching the exports.
	 *
	 * @return void
	 */
	public function handle_ajax_fetch_exports(): void {
		if ( ! learndash_is_group_leader_user() && ! learndash_is_admin_user() ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_permission',
				)
			);
		}
		$nonce = sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'ld-essay-nonce' ) ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_nonce',
				)
			);
		}
		$user_id    = get_current_user_id();
		$transients = get_transient( $this->transient_key );
		if ( empty( $transients ) ) {
			wp_send_json_success(
				array(
					'exports' => array(),
				)
			);
		}
		$users_exports = array_keys(
			array_filter(
				$transients,
				static function ( $transient ) use ( $user_id ) {
					return $transient === $user_id;
				},
			)
		);
		if ( empty( $users_exports ) ) {
			wp_send_json_success(
				array(
					'exports' => array(),
				)
			);
		}
		$exports = array();
		foreach ( $users_exports as $user_exports ) {
			$processing_data = get_transient( $this->data_prefix . $user_exports );
			if ( empty( $processing_data ) || ! is_a( $processing_data, EssayExportProcessingData::class ) ) {
				continue;
			}
			if ( $processing_data->empty || ! $processing_data->is_done() ) {
				continue;
			}
			$exports[] = array(
				'processing_id'   => $processing_data->processing_id,
				'last_saved'      => $processing_data->get_last_saved(),
				'last_saved_unix' => $processing_data->get_last_saved( 'U' ),
				'start_date'      => $processing_data->start_date ? $processing_data->start_date->format( 'd.m.Y' ) : '',
				'end_date'        => $processing_data->end_date ? $processing_data->end_date->format( 'd.m.Y' ) : '',
				'count'           => $processing_data->total_posts,
			);
		}

		usort(
			$exports,
			static function ( $a, $b ) {
				return absint( $b['last_saved_unix'] ) - absint( $a['last_saved_unix'] );
			}
		);

		wp_send_json_success(
			array(
				'exports' => $exports,
			)
		);
	}
}
