<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase

namespace LD_Organization\Hooks;

use Exception;
use LD_Organization\Organization;
use Monolog\Logger;
use DateTime;

use function LD_Organization\send_email;

/**
 * Class UserActivityAjax
 *
 * @package LD_Organization\Hooks
 * @since 0.22.0
 */
class UserActivityAjax extends AbstractHook {
	/**
	 * Logger instance
	 *
	 * @var Logger
	 */
	private Logger $logger;

	/**
	 * Sets up the class
	 */
	public function __construct() {
		parent::__construct();
		$this->logger = logger( 'UserActivityAjax' );
	}

	/**
	 * @inheritDoc
	 */
	final public function get_actions(): array {
		return array(
			array(
				'hook'     => 'wp_ajax_fetch_org_courses',
				'callable' => array( $this, 'handle_fetching_org_courses' ),
				'priority' => 10,
				'num_args' => 0,
			),
			array(
				'hook'     => 'wp_ajax_fetch_org_course_activity',
				'callable' => array( $this, 'handle_fetching_org_course_activity' ),
				'priority' => 10,
				'num_args' => 0,
			),
			array(
				'hook'     => 'wp_ajax_reset_org_course_activity',
				'callable' => array( $this, 'handle_resetting_course_activity' ),
				'priority' => 10,
				'num_args' => 0,
			),
		);
	}

	/**
	 * @inheritDoc
	 */
	final public function get_filters(): array {
		return array();
	}

	/**
	 * Handles the fetching of organization courses ajax request.
	 *
	 * @return void
	 */
	final public function handle_fetching_org_courses(): void {
		if ( ! learndash_is_admin_user() ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_permission',
				)
			);
		}
		$nonce = sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'ld-organization-activity-nonce' ) ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_nonce',
				)
			);
		}

		$organization_id = absint( $_POST['organization_id'] ?? 0 );
		if ( 0 === $organization_id ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_organization_id',
				)
			);
		}
		try {
			$organization  = Organization::with_id( $organization_id );
			$groups        = $organization->get_organization_groups();
			$courses       = array();
			$found_courses = array();
			foreach ( $groups as $group_id ) {
				$course_list = learndash_get_group_courses_list( $group_id );
				if ( ! $course_list || empty( $course_list ) ) {
					continue;
				}
				foreach ( $course_list as $course_id ) {
					if ( in_array( $course_id, $found_courses, true ) ) {
						continue;
					}
					$found_courses[] = $course_id;
					$courses[]       = array(
						'id'    => $course_id,
						'title' => get_the_title( $course_id ),
					);
				}
			}
			if ( empty( $courses ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'No courses found', 'ld-organization' ),
					)
				);
			}
			wp_send_json_success( array( 'courses' => array_values( $courses ) ) );
		} catch ( Exception $e ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_organization',
				)
			);
		}
	}

	/**
	 * Handles the fetching of organization course activity ajax request.
	 *
	 * @return void
	 */
	final public function handle_fetching_org_course_activity(): void {
		if ( ! learndash_is_admin_user() ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_permission',
				)
			);
		}
		$nonce = sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'ld-organization-course-activity-nonce' ) ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_nonce',
				)
			);
		}
		$organization_id = absint( $_POST['organization_id'] ?? 0 );
		if ( 0 === $organization_id ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_organization_id',
				)
			);
		}
		$course_id = absint( $_POST['course_id'] ?? 0 );
		if ( 0 === $course_id ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_course_id',
				)
			);
		}
		try {
			$organization = Organization::with_id( $organization_id );
			$groups       = $organization->get_organization_groups();
			$activity     = array();
			$users        = array();
			foreach ( $groups as $group_id ) {
				$group_users = learndash_get_groups_users( $group_id );
				foreach ( $group_users as $user ) {
					if ( isset( $users[ $user->ID ] ) ) {
						continue;
					}
					$users[ $user->ID ] = $user;
				}
			}
			$comparison_date = new DateTime();
			$comparison_date->modify( '-1005 days' ); // 2 years and ~9 months ago
			foreach ( $users as $user_id => $user ) {
				$completed_date = learndash_user_get_course_completed_date( $user_id, $course_id );
				$has_been_reset = get_user_meta( $user_id, 'ld_organization_course_activity_reset_' . $course_id, true );

				try {
					if ( ! empty( $has_been_reset ) ) {
						$reset_date = DateTime::createFromFormat( 'U', $has_been_reset );
						$reset_date = $reset_date->format( 'Y-m-d H:i:s' );
					} else {
						$reset_date = '-';
					}
				} catch ( Exception $e ) {
					$reset_date = '-';
				}

				$is_threshold = false;
				try {
					$completed_date = ( $completed_date && $completed_date !== '0' && $completed_date !== '' ) ? DateTime::createFromFormat( 'U', $completed_date ) : '-';
					if ( $completed_date instanceof DateTime ) {
						$is_threshold   = $completed_date->getTimestamp() < $comparison_date->getTimestamp();
						$completed_date = $completed_date->format( 'Y-m-d H:i:s' );
					}
				} catch ( Exception $e ) {
					$completed_date = '-';
				}
				$activity[] = array(
					'user_id'        => $user->ID,
					'user_name'      => $user->display_name,
					'user_email'     => $user->user_email,
					'completed'      => $completed_date !== '-',
					'completed_date' => $completed_date,
					'is_threshold'   => $is_threshold,
					'reset_date'     => $reset_date,
				);
			}
			wp_send_json_success(
				array(
					'message' => 'Organization course activity fetched successfully',
					'users'   => $activity,
				)
			);
		} catch ( Exception $e ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_organization',
				)
			);
		}
		wp_send_json_error( array( 'message' => 'Error fetching organization course activity' ) );
	}

	/**
	 * Handles the resetting of course activity ajax request.
	 *
	 * @return void
	 */
	public function handle_resetting_course_activity(): void {
		if ( ! learndash_is_admin_user() ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_permission',
				)
			);
		}
		$nonce = sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'ld-organization-course-reset-activity-nonce' ) ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_nonce',
				)
			);
		}
		$organization_id = absint( $_POST['organization_id'] ?? 0 );
		if ( 0 === $organization_id ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_organization_id',
				)
			);
		}
		try {
			$_organization = Organization::with_id( $organization_id );
		} catch ( Exception $e ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_organization',
				)
			);
		}
		$course_id = absint( $_POST['course_id'] ?? 0 );
		if ( 0 === $course_id ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_course_id',
				)
			);
		}
		if ( ! learndash_is_course_post( $course_id ) ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_course_id',
				)
			);
		}
		$raw_user_ids = $_POST['user_ids'];
		if ( is_null( $raw_user_ids ) || ! is_string( $raw_user_ids ) ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_user_ids',
				)
			);
		}
		$raw_user_ids = array_map( 'trim', explode( ',', $raw_user_ids ) );
		$user_ids     = array_map( 'absint', $raw_user_ids );
		if ( empty( $user_ids ) ) {
			wp_send_json_error(
				array(
					'error' => 'invalid_user_id',
				)
			);
		}

		$quizzes = get_posts(
			array(
				'post_type'      => 'sfwd-quiz',
				'posts_per_page' => -1,
				'fields'         => 'ids',
				'meta_query'     => array(
					'relation' => 'AND',
					array(
						'key'     => 'course_id',
						'value'   => $course_id,
						'compare' => '=',
					),
					array(
						'key'     => 'reset_progress',
						'value'   => '1',
						'compare' => '=',
					),
				),
			)
		);

		if ( empty( $quizzes ) ) {
			wp_send_json_error(
				array(
					'error' => 'no_quizzes_found',
				)
			);
		}
		$update_array                       = array();
		$update_array['quiz']               = array();
		$update_array['quiz'][ $course_id ] = array();
		foreach ( $quizzes as $quiz_id ) {
			$update_array['quiz'][ $course_id ][ $quiz_id ] = '0';
		}

		$users_to_send = array();

		foreach ( $user_ids as $user_id ) {
			$user = get_user_by( 'id', $user_id );
			if ( ! $user ) {
				continue;
			}
			$user_activity = learndash_user_get_course_progress( $user_id, $course_id );
			if ( ! $user_activity || ! isset( $user_activity['status'] ) ) {
				continue;
			}
			if ( $user_activity['status'] !== 'completed' ) {
				continue;
			}
			$handled = learndash_process_user_course_progress_update( $user_id, $update_array );
			if ( empty( $handled ) ) {
				continue;
			}
			update_user_meta( $user_id, 'ld_organization_course_activity_reset_' . $course_id, time() );
			send_email(
				'Kädenjälki <noreply@kadenjalki.fi>',
				$user->user_email,
				array(
					'first_name'  => $user->first_name,
					'course_name' => get_the_title( $course_id ),
				),
				'user-progress-reset'
			);
		}

		wp_send_json_success(
			array(
				'message' => 'Course activity reset successfully',
			)
		);
	}
}
