<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
// phpcs:disable Generic.Commenting.DocComment.MissingShort
/**
 * Handles Polylang for the REST API.
 *
 * @package LD_Organization\Hooks
 */

namespace LD_Organization\Hooks;

use Exception;
use PLL_Base;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Correctly sets up the Polylang current language for use in the REST API.
 *
 * @package LD_Organization\Hooks
 */
class PolylangRestAPI extends AbstractHook {

	/**
	 * Handles setting the current language for polylang.
	 *
	 * @return void
	 */
	final public function set_current_language(): void {
		global $polylang;

		$default             = pll_default_language();
		$available_languages = pll_languages_list();

		if ( isset( $_SERVER['HTTP_X_REST_LANGUAGE'] ) && is_string( $_SERVER['HTTP_X_REST_LANGUAGE'] ) ) {
			$cur_lang = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_REST_LANGUAGE'] ) );
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: We are not processing form information.
		if ( isset( $_GET['lang'] ) && is_string( $_GET['lang'] ) ) {
			// Fix up the GET variable.
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: We are not processing form information.
			$cur_lang = sanitize_text_field( wp_unslash( $_GET['lang'] ) );
		}

		// Set the current language to the requested language or default.
		if ( empty( $cur_lang ) || ! in_array( $cur_lang, $available_languages, true ) ) {
			$cur_lang = $default;
		}

		$polylang->curlang = $polylang->model->get_language( $cur_lang );

		$current_languages = array(
			'fi' => 'fi',
			'en' => 'en_US',
		);
		$locale            = $current_languages[ $cur_lang ] ?? 'fi';

		// This action is here if you need to unload some textdomains before switching.
		do_action( 'kadenjalki_before_locale_switch', $cur_lang, $current_languages[ $cur_lang ] ?? 'fi' );
		unload_textdomain( 'ld-organization' );
		// Set the locale.
		$locale = $current_languages[ $cur_lang ] ?? 'fi';
		switch_to_locale( $locale );
		// This action is here if you need to load some textdomains after switching.
		do_action( 'kadenjalki_after_locale_switch', $cur_lang, $current_languages[ $cur_lang ] ?? 'fi' );
		i18nize();

		// Ugly ugly hack to get the correct textdomain..
		// Ultra ugly to get the correct string translations..
		if ( class_exists( 'PLL_Base' ) && is_a( $polylang, PLL_Base::class ) ) {
			try {
				unload_textdomain( 'pll_string' );
				$polylang->load_strings_translations( $locale );
			} catch ( Exception $e ) {
				// Ignore failures.
			}
		}
	}

	/**
	 * @inheritDoc
	 */
	final public function get_actions(): array {
		return array(
			array(
				'hook'     => 'rest_api_init',
				'callable' => array( $this, 'set_current_language' ),
				'priority' => 10,
				'num_args' => 0,
			),
		);
	}

	/**
	 * @inheritDoc
	 */
	final public function get_filters(): array {
		return array();
	}
}
