<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
/**
 * Handles registering the RestAPI endpoints for handling organizations.
 *
 * @package LD_Organization
 */

namespace LD_Organization\Rest;

use InvalidArgumentException;
use LD_Organization\Exceptions\OrganizationNotFoundException;
use LD_Organization\Exceptions\PermissionException;
use LD_Organization\TransientManager;
use LD_Organization\Organization;
use LDLMS_Model_Course;
use LDLMS_Quiz_Questions;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use WpProQuiz_Model_AnswerTypes;
use WpProQuiz_Model_CategoryMapper;
use WpProQuiz_Model_Question;
use function LD_Organization\calculate_groups_course_progress;
use function LD_Organization\fetch_quiz_statistics;
use function LD_Organization\get_default_image_url;
use function LD_Organization\get_signed_enrollment_url;
use function LD_Organization\get_users_organization_id;
use function LD_Organization\handle_erroneous_values;
use function LD_Organization\is_quiz_attempt_passed;
use function LD_Organization\send_email;
use function LD_Organization\user_has_role;

/**
 * RestAPI class, when initialized will register the RestAPI.
 *
 * @package LD_Organization
 * @since 0.2.0
 */
class RestAPI {

	/**
	 * The RestAPI namespace.
	 *
	 * @var string
	 */
	private string $namespace = 'ld-organization';

	/**
	 * The RestAPI version.
	 *
	 * @var string
	 */
	private string $version = 'v1';

	/**
	 * Registers all required functionality and endpoints.
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_endpoints' ) );
	}

	/**
	 * Registers all the endpoint and their parameters.
	 *
	 * @return void
	 */
	final public function register_endpoints(): void {
		$this->register_organization_routes();
		$this->register_groups_routes();
		$this->register_user_routes();
	}

	/**
	 * Gets data of the courses in requested organization
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_organization_courses( WP_REST_Request $request ): WP_REST_Response {
		try {
			$org_id       = $request->get_param( 'id' );
			$organization = Organization::with_id( $org_id );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$transient_name = 'org_' . $org_id . '_course_data';

			$data = array();

			$groups = $organization->get_organization_groups();

			foreach ( $groups as $group_id ) {

				if ( ! $group_id ) {
					continue;
				}

				$courses = learndash_get_group_courses_list( $group_id );

				foreach ( $courses as $course_id ) {

					if ( ! $course_id ) {
						continue;
					}

					$course = get_post( $course_id );

					// Excludes admins.
					$enrolled_users_query = learndash_get_users_for_course( $course_id, array(), true );

					$enrolled_users_id = $enrolled_users_query->get_results();

					$enrolled_users = count( $enrolled_users_id );

					$course_completions = 0;

					foreach ( $enrolled_users_id as $enrolled_user_id ) {
						$progress = learndash_user_get_course_progress( $enrolled_user_id, $course_id );
						if ( $progress['completed'] === $progress['total'] ) {
							++ $course_completions;
						}
					}

					$data[] = array(
						'course_id'   => $course_id,
						'title'       => $course->post_title,
						'enrolled'    => $enrolled_users,
						'enrolled_id' => $enrolled_users_id,
						'completions' => $course_completions,
					);

				}
			}

			if ( ! empty( $data ) ) {
				TransientManager::set( $transient_name, $data );
			}

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $data,
					'message' => 'Successfully fetched courses for organization ' . $org_id,
				)
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'Organization not found', 'ld-organization' ),
				),
				404
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Gets the requested users courses and the progress for the requested group.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_user_group_progress( WP_REST_Request $request ): WP_REST_Response {
		try {
			$group_id          = $request->get_param( 'group_id' );
			$requested_user_id = $request->get_param( 'user_id' );

			$transient_name = 'users_group_' . $group_id . '_courses_' . $requested_user_id;

			if ( false !== ( $value = TransientManager::get( $transient_name ) ) ) {
				return new WP_REST_Response(
					array(
						'success' => true,
						'data'    => $value,
					)
				);
			}

			$courses = learndash_get_group_courses_list( $group_id );

			$data = array();

			// Loops over each course
			foreach ( $courses as $course_id ) {
				$course          = get_post( $course_id );
				$course_progress = learndash_user_get_course_progress( $requested_user_id, $course_id );

				$lessons      = learndash_get_course_lessons_list( $course_id, $requested_user_id );
				$lessons_data = array();

				// Loops over each lesson.
				foreach ( $lessons as $lesson ) {

					if ( ! $lesson ) {
						continue;
					}

					$lesson_data = $lesson['post'];

					// Gets topics of lesson.
					$topics                    = learndash_course_get_topics( $course_id, $lesson_data->ID );
					$topics_data               = array();
					$lesson_progress_topics    = $course_progress['topics'][ $lesson_data->ID ];
					$lesson_progress_total     = count( $lesson_progress_topics );
					$lesson_progress_completed = array_reduce(
						array_keys( $lesson_progress_topics ),
						static function ( $carry, $key ) use ( $lesson_progress_topics ) {
							return $carry + $lesson_progress_topics[ $key ];
						}
					);

					// Calculate percentage of lesson progress.
					$progress_percentage_lesson = 0;
					if ( $lesson_progress_completed !== 0 && $lesson_progress_total !== 0 && $lesson_progress_completed <= $lesson_progress_total ) {
						$progress_percentage_lesson = handle_erroneous_values( ( $lesson_progress_completed / $lesson_progress_total ) * 100 );
						$progress_percentage_lesson = round( $progress_percentage_lesson );
					}

					foreach ( $topics as $topic ) {
						if ( ! $topic ) {
							continue;
						}

						$topic_completed = 1 === $lesson_progress_topics[ $topic->ID ];

						$topics_data[] = array(
							'title'     => $topic->post_title,
							'id'        => $topic->ID,
							'completed' => $topic_completed,
						);

					}

					$lessons_data[] = array(
						'id'                            => $lesson_data->ID,
						'title'                         => $lesson_data->post_title,
						'topics'                        => $topics_data,
						'progress_completed_percentage' => $progress_percentage_lesson,
					);
				}

				$quizzes      = learndash_get_course_quiz_list( $course_id );
				$quizzes_data = array();

				// Loops over each quiz.
				foreach ( $quizzes as $quiz ) {

					if ( ! $quiz ) {
						continue;
					}

					$quiz_data      = $quiz['post'];
					$quiz_completed = learndash_user_quiz_has_completed( $requested_user_id, $quiz_data->ID, $course_id );

					$quizzes_data[] = array(
						'id'        => $quiz_data->ID,
						'title'     => $quiz_data->post_title,
						'completed' => $quiz_completed,
					);

				}

				if ( ! $course ) {
					continue;
				}

				// Calculate percentage of course progress.
				$progress_percentage = 0;
				if ( $course_progress['completed'] !== 0 && $course_progress['total'] !== 0 && $course_progress['completed'] <= $course_progress['total'] ) {
					$progress_percentage = handle_erroneous_values( ( $course_progress['completed'] / $course_progress['total'] ) * 100 );
					$progress_percentage = round( $progress_percentage );
				}

				$data[] = array(
					'id'                            => $course->ID,
					'title'                         => $course->post_title,
					'progress_completed_percentage' => $progress_percentage,
					'lessons'                       => $lessons_data,
					'quizzes'                       => $quizzes_data,
				);
			}

			if ( ! empty( $data ) ) {
				TransientManager::set( $transient_name, $data );
			}

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $data,
				)
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Handles fetching all of the users certificates and returns the information.
	 *
	 * @param WP_REST_Request $request The request object on the current request.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_user_certificates( WP_REST_Request $request ): WP_REST_Response {
		try {
			$requested_user_id = $request->get_param( 'user_id' );

			$transient_name = 'user_certificates_' . $requested_user_id;

			if ( false !== ( $value = TransientManager::get( $transient_name ) ) ) { // phpcs:ignore WordPress.CodeAnalysis.AssignmentInCondition.Found,Squiz.PHP.DisallowMultipleAssignments.FoundInControlStructure
				return new WP_REST_Response(
					array(
						'success' => true,
						'data'    => $value,
					)
				);
			}

			$course_ids = learndash_user_get_enrolled_courses( $requested_user_id, array(), true );
			$quizzes    = get_user_meta( $requested_user_id, '_sfwd-quizzes', true );

			$data = array();

			// Course certificates handled here.
			if ( ! empty( $course_ids ) ) {
				$title = '';
				foreach ( $course_ids as $course_id ) {

					$certificate_link = learndash_get_course_certificate_link( $course_id, $requested_user_id );

					if ( ! empty( $certificate_link ) ) {
						if ( empty( $title ) ) {
							$post = get_post( $course_id );
							if ( ! empty( $post ) ) {
								$title = $post->post_title;
							}
						}
						$data[] = array(
							'title' => $title,
							'link'  => $certificate_link,
							'type'  => 'course',
						);
					}
				}
			}

			// Quiz certificates handled here.
			$already_added = array();

			if ( ! empty( $quizzes ) ) {
				foreach ( $quizzes as $quiz_attempt ) {
					if ( empty( $quiz_attempt['quiz'] ) ) {
						continue;
					}
					if ( isset( $already_added[ $quiz_attempt['quiz'] ] ) ) {
						continue;
					}
					$certificate = learndash_certificate_details( $quiz_attempt['quiz'], $requested_user_id );
					$title       = '';
					if ( isset( $certificate['certificateLink'] ) ) {
						$already_added[ $quiz_attempt['quiz'] ] = true;
						$post                                   = get_post( $quiz_attempt['quiz'] );
						if ( ! empty( $post ) ) {
							$title = $post->post_title;
						}

						$data[] = array(
							'title' => $title,
							'link'  => $certificate['certificateLink'],
							'type'  => 'quiz',
						);
					}
				}
			}

			// Group certificates handled here.
			$user_groups = learndash_get_users_group_ids( $requested_user_id );

			if ( ! empty( $user_groups ) ) {
				foreach ( $user_groups as $group_id ) {
					$certificate_link = learndash_get_group_certificate_link( $group_id, $requested_user_id );
					$title            = '';
					if ( ! empty( $certificate_link ) ) {
						$post = get_post( $group_id );
						if ( ! empty( $post ) ) {
							$title = $post->post_title;
						}
						$data[] = array(
							'title' => $title,
							'link'  => $certificate_link,
							'type'  => 'group',
						);
					}
				}
			}

			if ( ! empty( $data ) ) {
				TransientManager::set( $transient_name, $data, 2 * MINUTE_IN_SECONDS );
			}

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $data,
				)
			);

		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );

	}

	/**
	 * Handles returning a list of groups for the requested users ID.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_users_groups( WP_REST_Request $request ): WP_REST_Response {
		try {
			$requested_user_id = $request->get_param( 'user_id' );

			$transient_name = 'user_groups_' . $requested_user_id;

			if ( false !== ( $value = TransientManager::get( $transient_name ) ) ) {
				return new WP_REST_Response(
					array(
						'success' => true,
						'data'    => $value,
					)
				);
			}

			$groups = learndash_get_users_group_ids( $requested_user_id );

			$data = array();
			// Loop through all groups and create a smaller data array.
			foreach ( $groups as $group_id ) {
				$group = get_post( $group_id );
				if ( ! $group ) {
					continue;
				}
				$data[] = array(
					'id'    => $group->ID,
					'title' => $group->post_title,
				);
			}

			if ( ! empty( $data ) ) {
				TransientManager::set( $transient_name, $data );
			}

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $data,
				)
			);

		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Fetches the requested organization and their allowed courses, if a school, otherwise it returns all courses.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 * @since 0.5.0
	 */
	final public function get_organization_allowed_courses( WP_REST_Request $request ): WP_REST_Response {
		try {
			$organization = Organization::with_id( $request->get_param( 'id' ) );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$course_ids       = $organization->get_allowed_courses_ids();
			$group_id         = $request->get_param( 'group_id' );
			$group_course_ids = learndash_get_groups_courses_ids( get_current_user_id(), array( $group_id ) );
			$final_course_ids = array_filter(
				$course_ids,
				static function ( $course_id ) use ( $group_course_ids ) {
					return ! in_array( $course_id, $group_course_ids, true );
				}
			);

			$response = new WP_REST_Response(
				array(
					'success'    => true,
					'course_ids' => array_values( $final_course_ids ),
				),
				200
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				404
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Returns the organizations groups enrollment URL.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_group_enrollment_url( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$org_id   = $request->get_param( 'id' );

		$transient_name = $org_id . '_' . $group_id . '_enrollment_url';

		if ( false !== ( $value = TransientManager::get( $transient_name ) ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $value,
				),
				200
			);
		}
		try {
			$organization = Organization::with_id( $org_id );
			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$enrollment_url = get_signed_enrollment_url( $group_id );

			$data = array( 'enrollment_url' => $enrollment_url );

			TransientManager::set( $transient_name, $data );

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $data,
				),
				200,
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found.',
				),
				404
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Gets all users and their progress for the requested group.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_groups_user( WP_REST_Request $request ): WP_REST_Response {
		$group_id  = $request->get_param( 'group_id' );
		$course_id = $request->get_param( 'course_id' );

		$transient_name = $group_id . '_group_users_progress';

		if ( ! empty( $course_id ) ) {
			$transient_name .= '_course_' . $course_id;
		}

		if ( false !== ( $value = TransientManager::get( $transient_name ) ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $value,
				)
			);
		}

		$user_ids = learndash_get_groups_user_ids( $group_id, true );

		$users = array();

		foreach ( $user_ids as $user_id ) {
			$user      = get_userdata( $user_id );
			$user_data = array(
				'user_id'    => $user->ID,
				'first_name' => $user->first_name,
				'last_name'  => $user->last_name,
			);
			if ( ! empty( $course_id ) ) {
				$progress     = learndash_user_get_course_progress( $user_id, $course_id, 'summary' );
				$completed_on = learndash_user_get_course_completed_date( $user_id, $course_id );
				if ( ! empty( $completed_on ) ) {
					$progress['completed_on'] = $completed_on;
				}
				if ( ! empty( $progress['completed'] ) && ! empty( $progress['total'] ) ) {
					$progress['percentage'] = ceil( ( $progress['completed'] / $progress['total'] ) * 100 );
				}

				$status = learndash_course_status( $course_id, $user_id );
			} else {
				$progress = learndash_get_user_group_progress( $group_id, $user_id, true );
				$status   = learndash_get_user_group_status( $group_id, $user_id );
			}
			$users[] = array_merge( $user_data, $progress, array( 'status' => $status ) );
		}

		if ( ! empty( $users ) ) {
			TransientManager::set( $transient_name, $users, 5 * MINUTE_IN_SECONDS );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $users,
			)
		);
	}

	/**
	 * Handles fetching a list of group leaders for the current group.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_group_leaders( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );

		$transient_name = $group_id . 'group_group_leaders';

		if ( false !== ( $value = TransientManager::get( $transient_name ) ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $value,
				)
			);
		}

		$group_leaders = learndash_get_groups_administrator_ids( $group_id, true );

		$users = array();

		foreach ( $group_leaders as $user_id ) {
			$user      = get_userdata( $user_id );
			$user_data = array(
				'user_id'    => $user->ID,
				'first_name' => $user->first_name,
				'last_name'  => $user->last_name,
				'email'      => $user->user_email,
			);
			$users[]   = $user_data;
		}

		if ( ! empty( $users ) ) {
			TransientManager::set( $transient_name, $users, 5 * MINUTE_IN_SECONDS );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $users,
			)
		);
	}

	/**
	 * Handles getting a list of quizzes for the requested course.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_course_quizzes( WP_REST_Request $request ): WP_REST_Response {
		$course_id = $request->get_param( 'course_id' );

		$transient_name = '_course_quiz_' . $course_id;

		if ( false !== ( $value = TransientManager::get( $transient_name ) ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $value,
				)
			);
		}

		$steps = learndash_course_get_steps_by_type( $course_id, learndash_get_post_type_slug( 'quiz' ) );

		$final_list = array();
		foreach ( $steps as $item ) {
			$post = get_post( $item );
			if ( ! empty( $post ) ) {
				$final_list[] = array(
					'title' => $post->post_title,
					'id'    => $post->ID,
				);
			}
		}

		if ( ! empty( $final_list ) ) {
			TransientManager::set( $transient_name, $final_list, 5 * MINUTE_IN_SECONDS );
		} else {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'Course does not have any assigned quiz.', 'ld-organization' ),
				),
				404
			);
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $final_list,
			),
			200
		);

	}

	/**
	 * Handles fetching the information for a group course quizzes.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_group_course_quiz( WP_REST_Request $request ): WP_REST_Response {
		$group_id  = $request->get_param( 'group_id' );
		$course_id = $request->get_param( 'course_id' );
		$quiz_id   = $request->get_param( 'quiz_id' );

		$group_users = learndash_get_groups_user_ids( $group_id );

		if ( empty( $group_users ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => array(),
				),
				200
			);
		}

		$final_progress = array();

		// Fetches the users quiz attempts and filters to get only the completed quiz attempts.
		foreach ( $group_users as $user_id ) {
			$attempt = learndash_get_user_quiz_attempt(
				$user_id,
				array(
					'quiz'   => $quiz_id,
					'course' => $course_id,
				)
			);

			// Since the previous function can return "" we need to filter out that first.
			if ( empty( $attempt ) || ! is_array( $attempt ) ) {
				continue;
			}

			// Adds a custom is_passed check.
			foreach ( $attempt as $key => $att ) {
				$attempt[ $key ]['has_passed'] = is_quiz_attempt_passed( $att );
			}

			$attempt = array_filter(
				$attempt,
				static function( $a ) {
					return isset( $a['completed'] );
				}
			);

			// Now we can filter out empty arrays.
			if ( empty( $attempt ) ) {
				continue;
			}

			$user = get_user_by( 'id', $user_id );

			$user_data = array(
				'user_id'    => $user_id,
				'first_name' => $user->first_name,
				'last_name'  => $user->last_name,
			);

			$final_progress[] = array_merge( $user_data, array( 'attempts' => array_values( $attempt ) ) );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $final_progress,
			),
			200
		);

	}

	/**
	 * Handles the retrieval of a users quiz attempts answer and questions.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_quiz_data( WP_REST_Request $request ): WP_REST_Response {
		$group_id  = $request->get_param( 'group_id' );
		$quiz_id   = $request->get_param( 'quiz_id' );
		$user_id   = $request->get_param( 'user_id' );
		$course_id = $request->get_param( 'course_id' );

		// LDLMS_Quiz_Questions is a model that can be used to get questions, the parameter pro_objects, makes the questions a better format than a pure WordPress post.
		$question_model = new LDLMS_Quiz_Questions( $quiz_id );
		$questions      = $question_model->get_questions( 'pro_objects' );

		$category_mapper = new WpProQuiz_Model_CategoryMapper();

		$question_data = array();

		// We use assert because we need to be sure that the get_questions call always returns what is expected.
		foreach ( $questions as $question ) {
			assert( $question instanceof WpProQuiz_Model_Question );
			$question_array = $question->get_object_as_array();

			if ( ! empty( $question->getCategoryId() ) ) {
				$question_array['_categoryId'] = $question->getCategoryId();
				$category                      = $category_mapper->fetchById( $question->getCategoryId() );
				if ( $category->getCategoryId() === $question->getCategoryId() ) { // We check to be sure that we got the right category (or any category for that matter).
					$question_array['_categoryName'] = $category->getCategoryName();
				}
			}

			foreach ( $question_array['_answerData'] as $key => $answer_data ) {
				assert( $answer_data instanceof WpProQuiz_Model_AnswerTypes );
				$question_array['_answerData'][ $key ] = $answer_data->get_object_as_array();
				if ( $question->getAnswerType() === 'cloze_answer' ) {
					$question_array['_answerData'][ $key ]['converted_answer'] = learndash_question_cloze_fetch_data( $answer_data->getAnswer() );
					$question_array['_answerData'][ $key ]['prepared_output']  = learndash_question_cloze_prepare_output( $question_array['_answerData'][ $key ]['converted_answer'] );
				}
				if ( $question->getAnswerType() === 'free_answer' ) {
					$question_array['_answerData'][ $key ]['converted_answer'] = learndash_question_free_get_answer_data( $answer_data );
				}
			}
			$question_data[] = $question_array;
		}

		$attempts = learndash_get_user_quiz_attempt(
			$user_id,
			array(
				'quiz'   => $quiz_id,
				'course' => $course_id,
			)
		);

		if ( ! is_array( $attempts ) ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => __(
						'No attempts for this quiz found',
						'ld-organization'
					),
				),
				404
			);
		}

		foreach ( $attempts as $key => $attempt ) {
			if ( isset( $attempt['statistic_ref_id'], $attempt['pro_quizid'] ) ) {
				$stats                            = fetch_quiz_statistics( (int) $attempt['statistic_ref_id'], $attempt['pro_quizid'] );
				$attempts[ $key ]['answer_stats'] = $stats;
			}
		}

		$answer_data = $attempts;

		$data = array(
			'success' => true,
			'data'    => array(
				'questions' => $question_data,
				'answers'   => $answer_data,
			),
		);

		return new WP_REST_Response( $data, 200 );
	}

	/**
	 * Handles fetching the organization based on the current user.
	 *
	 * @param WP_REST_Request $request The current request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_organization( WP_REST_Request $request ): WP_REST_Response {
		try {
			$user_id      = get_current_user_id();
			$organization = Organization::by_user_id( $user_id );
			$post         = $organization->get_organization();
			if ( null !== $post ) {
				$organization_data = array(
					'success' => true,
					'data'    => array(
						'id'   => $post->ID,
						'name' => $post->post_title,
						'slug' => $post->post_name,
						'type' => $organization->get_type(),
					),
				);
			} else {
				throw new OrganizationNotFoundException();
			}
			return new WP_REST_Response( $organization_data, 200 );
		} catch ( OrganizationNotFoundException $e ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found.',
				),
				404
			);
		}
	}

	/**
	 * Handles fetching users that match search string
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_REST_Response
	 */
	final public function search_user( WP_REST_Request $request ): WP_REST_Response {
		try {
			$search_string   = strtolower( $request->get_param( 'searchString' ) );
			$current_user_id = get_current_user_id();
			$organization    = Organization::by_user_id( $current_user_id );

			$allowed_roles = array( 'opettaja', 'administrator', 'group_leader' );
			if ( ! is_user_logged_in() && empty( array_intersect( $allowed_roles, wp_get_current_user()->roles ) ) ) {
				$response = new WP_REST_Response(
					array(
						'success' => false,
						'message' => 'User does not have the authority to do this.',
					),
					404
				);
				return rest_ensure_response( $response );
			}

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$data        = array();
			$user_ids    = learndash_get_groups_administrators_users( $current_user_id );
			$users_exist = array();

			$users = array();

			foreach ( $user_ids as $user_id ) {

				if ( ! $user_id || in_array( $user_id, $users_exist, true ) ) {
					continue;
				}

				$users_exist[] = $user_id;

				$user_meta = get_userdata( $user_id );

				$first_name      = strtolower( $user_meta->user_firstname );
				$last_name       = strtolower( $user_meta->user_lastname );
				$first_last_name = strtolower( $user_meta->user_firstname . ' ' . $user_meta->user_lastname );
				$display_name    = strtolower( $user_meta->display_name );

				if (
					strpos( $first_name, $search_string ) !== false ||
					strpos( $last_name, $search_string ) !== false ||
					strpos( $first_last_name, $search_string ) !== false ||
					strpos( $display_name, $search_string ) !== false
				) {

					$users[] = array(
						'ID'           => $user_id,
						'firstname'    => $first_name,
						'lastname'     => $last_name,
						'first_last'   => $first_last_name,
						'display_name' => $display_name,
						'exact_match'  => true,
						'levenshtein'  => -1,
					);

				} else {

					$leven_first      = levenshtein( $search_string, $first_name, 2, 3, 3 );
					$leven_last       = levenshtein( $search_string, $last_name, 2, 3, 3 );
					$leven_first_last = levenshtein( $search_string, $first_last_name, 2, 3, 3 );
					$leven_display    = levenshtein( $search_string, $display_name, 2, 3, 3 );
					$leven_min        = min( $leven_first, $leven_last, $leven_first_last, $leven_display );

					$users[] = array(
						'ID'           => $user_meta->ID,
						'firstname'    => $first_name,
						'lastname'     => $last_name,
						'first_last'   => $first_last_name,
						'display_name' => $display_name,
						'exact_match'  => false,
						'levenshtein'  => $leven_min,
					);

				}
			}

			foreach ( $users as $user ) {

				if ( ! $user ) {
					continue;
				}

				if ( count( $data ) > 4 ) {
					break;
				}

				if ( $user['levenshtein'] <= 6 ) {
					$data[] = $user;
				}
			}

			usort(
				$data,
				static function( $a, $b ) {
					return strcmp( $a['levenshtein'], $b['levenshtein'] );
				}
			);

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $data,
					'message' => 'Successfully searched for users like ' . $search_string,
				)
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'Organization not found', 'ld-organization' ),
				),
				404
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}


	/**
	 * Returns the number of groups in the requested organization.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response The response object.
	 */
	final public function get_groups( WP_REST_Request $request ): WP_REST_Response {
		try {
			$id           = $request->get_param( 'id' );
			$organization = Organization::with_id( $id );

			$groups_id = $organization->get_organization_groups();
			$users     = array();
			$groups    = array();

			foreach ( $groups_id as $group_id ) {
				if ( ! $group_id ) {
					continue;
				}

				$group = get_post( $group_id );

				$users[] = learndash_get_groups_user_ids( $group_id );

				$groups[] = array(
					'id'    => $group->ID,
					'title' => $group->post_title,
				);

			}

			function flatten( array $array ): array {
				$return = array();
				array_walk_recursive(
					$array,
					function( $a ) use ( &$return ) {
						$return[] = $a;
					}
				);
				return $return;
			}

			$users    = flatten( $users );
			$users    = array_unique( $users );
			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => array(
						'groups' => $groups,
						'users'  => $users,
					),
					'message' => 'Successfully fetched groups for organization' . $id,
				)
			);
		} catch ( InvalidArgumentException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $exception->getMessage(),
				),
				400
			);
		} catch ( OrganizationNotFoundException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				404
			);
		}
		return rest_ensure_response( $response );
	}

	/**
	 * Returns the requested organizations licenses.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @uses \LD_Organization\Organization::with_id(),\LD_Organization\Organization::get_licenses(),\LD_Organization\Organization::get_used_licenses()
	 * @return WP_REST_Response
	 */
	final public function get_licenses( WP_REST_Request $request ): WP_REST_Response {
		try {
			$id           = $request->get_param( 'id' );
			$organization = Organization::with_id( $id );

			$used_licenses      = $organization->get_used_licenses();
			$total_licenses     = $organization->get_licenses();
			$remaining_licenses = $organization->get_remaining_licenses();

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => array(
						'used_licenses'      => $used_licenses,
						'total_licenses'     => $total_licenses,
						'remaining_licenses' => $remaining_licenses,
					),
					'message' => 'Successfully fetched licenses for organization' . $id,
				)
			);

		} catch ( InvalidArgumentException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $exception->getMessage(),
				),
				400
			);
		} catch ( OrganizationNotFoundException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				404
			);
		}
		return rest_ensure_response( $response );
	}

	/**
	 * Creates an extended information for a course based on the requesting group.
	 *
	 * @param WP_REST_Request $request The request object of the current WP_REST_Request instance.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_extended_course_information( WP_REST_Request $request ): WP_REST_Response {
		$course_id = (int) $request->get_param( 'course_id' );
		$group_id  = (int) $request->get_param( 'group_id' );

		$transient_name = 'extended_information_' . $group_id . '_' . $course_id;

		if ( false !== ( $value = TransientManager::get( $transient_name ) ) ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $value,
				),
				200
			);
		}

		/** The course object @var LDLMS_Model_Course|null $course */
		$course = \LDLMS_Factory_Post::course( $course_id );
		if ( null === $course ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => __(
						'Course not found',
						'ld-organization'
					),
				),
				400
			);
		}

		$groups_ids = learndash_get_course_groups( $course_id );

		if ( ! in_array( $group_id, $groups_ids, true ) ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'This group does not have access to this course', 'ld-organization' ),
				),
				403
			);
		}
		$extended_information = array();

		$extended_information['course_progress'] = calculate_groups_course_progress( $group_id, $course_id );
		$image_url                               = get_the_post_thumbnail_url( $course_id );
		if ( '' === $image_url || false === $image_url ) {
			$image_url = get_default_image_url( 'course', 'medium' );
		}
		$extended_information['image_url'] = $image_url;

		if ( ! empty( $extended_information ) ) {
			TransientManager::set( $transient_name, $extended_information, 5 * MINUTE_IN_SECONDS );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $extended_information,
			),
			200
		);

	}

	/**
	 * Can be used to fetch extra information about a group.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_extra_group_information( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$amount   = learndash_get_groups_user_ids( $group_id );
		$amount   = count( $amount );

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => array( 'user_amount' => $amount ),
			),
			200
		);
	}

	/**
	 * Gets the requesting users groups (depending on role param) and returns their ids.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_requesting_users_groups( WP_REST_Request $request ): WP_REST_Response {
		$type            = $request->get_param( 'role' );
		$current_user_id = get_current_user_id();
		// TODO: Maybe implement the other roles, if needed.
		if ( 'group_leader' === $type ) {
			$group_ids = learndash_get_administrators_group_ids( $current_user_id );
			$response  = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => $group_ids,
				)
			);
		}

		if ( ! isset( $response ) ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'No role specified',
				),
				400
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Returns a list of course IDs for the requested group.
	 *
	 * @param WP_REST_Request $request The current request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function get_groups_course_ids( WP_REST_Request $request ): WP_REST_Response {
		$group_id    = $request->get_param( 'group_id' );
		$courses_ids = learndash_get_groups_courses_ids( get_current_user_id(), array( $group_id ) );

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $courses_ids,
			)
		);
	}

	/**
	 * Handles adding course to organization.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 * @since 0.6.0
	 */
	final public function add_course_to_organization_group( WP_REST_Request $request ): WP_REST_Response {
		$group_id      = $request->get_param( 'group_id' );
		$org_id        = $request->get_param( 'id' );
		$new_course_id = $request->get_param( 'course_id' );

		try {
			$organization = Organization::with_id( $org_id );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$allowed_courses = $organization->get_allowed_courses_ids();
			if ( ! in_array( $new_course_id, $allowed_courses, true ) ) {
				$response = new WP_REST_Response(
					array(
						'success' => false,
						'message' => 'This course is not allowed to be added to this organization',
					),
					403
				);
			} else {

				$result = ld_update_course_group_access( $new_course_id, $group_id, false );
				if ( false === $result ) {
					$response = new WP_REST_Response(
						array(
							'success' => false,
							'message' => 'Failed to add course to group',
						),
						500
					);
				} else {
					$response = new WP_REST_Response(
						array(
							'success' => true,
							'message' => 'Added course to group',
						),
						201
					);
				}
			}
		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				404
			);
		}

		return rest_ensure_response( $response );

	}

	/**
	 * Handles sending an email to a user to sign up for a course.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function invite_user_to_group( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$org_id   = $request->get_param( 'id' );
		$email    = $request->get_param( 'email' );

		$email_rate_limit_transient = 'rate_limit_invitation_email_' . filter_var( $email, FILTER_SANITIZE_ENCODED );

		if ( false !== TransientManager::get( $email_rate_limit_transient ) ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'This user has already been invited.', 'ld-organization' ),
				),
				429
			);
		}

		try {
			$organization = Organization::with_id( $org_id );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			if ( $organization->get_remaining_licenses() <= 0 ) {
				return new WP_REST_Response(
					array(
						'success' => false,
						'message' => 'Organization is out of licenses',
					),
					403
				);
			}

			$existing_user = get_user_by( 'email', $email );

			$enrollment_url = get_signed_enrollment_url( $group_id );
			$group          = get_post( $group_id );

			$data = array(
				'enrollment_url' => $enrollment_url,
				'site_url'       => site_url(),
			);
			$data = array_merge( $data, array( 'group_name' => $group->post_title ) );

			if ( false !== $existing_user ) {
				$users_groups = learndash_get_users_group_ids( $existing_user->ID );
				if ( in_array( absint( $group_id ), $users_groups, true ) ) {
					return new WP_REST_Response(
						array(
							'success' => false,
							'message' => __(
								'User already in the group',
								'ld-organization'
							),
						),
						400
					);
				}

				$data['first_name'] = $existing_user->first_name;
				$success            = send_email( 'Kädenjälki <noreply@kadenjalki.fi>', $email, $data, 'group-invite-existing' );
			} else {
				$success = send_email( 'Kädenjälki <noreply@kadenjalki.fi>', $email, $data, 'group-invite-new' );
			}

			if ( $success ) {

				// If we get a success we stop the user from spam sending the email again to the same recipient.
				TransientManager::set( $email_rate_limit_transient, true, 12 * HOUR_IN_SECONDS );

				$response = new WP_REST_Response(
					array(
						'success' => true,
						'data'    => $data,
					),
					200,
				);
			} else {
				$response = new WP_REST_Response(
					array(
						'success' => false,
					),
					500
				);
			}
		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found.',
				),
				404
			);
		}

		return rest_ensure_response( $response );

	}

	/**
	 * Fetches a list of the available users for the requested groups.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 *
	 * @since 0.8.4
	 */
	final public function get_available_group_leaders( WP_REST_Request $request ): Wp_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$org_id   = $request->get_param( 'id' );

		try {
			$organization = Organization::with_id( $org_id );
			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			$group_leaders = $organization->get_group_leaders();

			$admin_ids = learndash_get_groups_administrator_ids( $group_id, true );

			$not_in_group = array_filter(
				$group_leaders,
				static function ( $leader ) use ( $admin_ids ) {
					return ( ! in_array( $leader->ID, $admin_ids, true ) );
				}
			);

			$available_leaders = array();

			foreach ( $not_in_group as $leader ) {
				$available_leaders[] = array(
					'id'    => $leader->ID,
					'name'  => $leader->first_name . ' ' . $leader->last_name,
					'email' => $leader->user_email,
				);
			}

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'message' => __( 'Fetched group leaders successfully', 'ld-organization' ),
					'data'    => $available_leaders,
				),
				200
			);

		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				404
			);
		}

		return rest_ensure_response( $response );

	}

	/**
	 * Adds a valid user to the group.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	final public function add_leader_to_group( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$org_id   = $request->get_param( 'id' );
		$user_id  = $request->get_param( 'user_id' );

		try {
			$organization = Organization::with_id( $org_id );
			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			if ( ! $organization->has_user( $user_id, 'group_leaders', true ) ) {
				throw new InvalidArgumentException( __( 'User is not in the organization', 'ld-organization' ) );
			}

			if ( ! user_has_role( $user_id, array( 'group_leader', 'opettaja', 'administrator' ) ) ) {
				throw new InvalidArgumentException( __( 'Selected user does not have the correct permissions', 'ld-organization' ) );
			}

			$admin_ids = learndash_get_groups_administrator_ids( $group_id, true );

			if ( in_array( $user_id, $admin_ids, true ) ) {
				throw new InvalidArgumentException( __( 'User already apart of the group.', 'ld-organization' ) );
			}

			$success = ld_update_leader_group_access( $user_id, $group_id );

			// If we managed to add the user, we can attempt to send an email to the user.
			if ( $success ) {
				$post       = get_post( $group_id );
				$group_name = $post->post_title;
				$login_url  = wp_login_url();
				$user       = get_user_by( 'ID', $user_id );

				$data = array(
					'group_name' => $group_name,
					'login_url'  => $login_url,
					'first_name' => $user->first_name,
				);

				if ( ! empty( $user->user_email ) ) {
					$success = send_email( 'Kädenjälki <noreply@kadenjalki.fi>', $user->user_email, $data, 'add-group-leader' );
				}
			}

			if ( $success ) {
				$response = new WP_REST_Response(
					array(
						'success' => true,
						'message' => __( 'User successfully added', 'ld-organization' ),
						'data'    => array(
							'email' => $success,
						),
					),
					201
				);

				TransientManager::delete( $group_id . 'group_group_leaders' );
			} else {
				$response = new WP_REST_Response(
					array(
						'success' => false,
						'message' => __( 'Unable to add user as group leader to this group', 'ld-organization' ),
					),
					500
				);
			}
		} catch ( OrganizationNotFoundException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				404
			);
		} catch ( InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				400
			);
		}
		return rest_ensure_response( $response );
	}

	/**
	 * Either creates or adds a user to the organization.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 * @noinspection PhpUnreachableStatementInspection
	 * @deprecated
	 */
	final public function invite_group_leader( WP_REST_Request $request ): WP_REST_Response {

		return new WP_REST_Response(
			array(
				'success' => false,
				'message' => __( 'This method has been deprecated', 'ld-organization' ),
			),
			404
		);
		// phpcs:disable Squiz.PHP.NonExecutableCode.Unreachable
		$group_id = $request->get_param( 'group_id' );
		$org_id   = $request->get_param( 'id' );

		$user_data = array(
			'email'      => $request->get_param( 'user_email' ),
			'user_email' => $request->get_param( 'user_email' ),
			'first_name' => $request->get_param( 'first_name' ),
			'last_name'  => $request->get_param( 'last_name' ),
		);

		try {
			$organization = Organization::with_id( $org_id );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			if ( ! $organization->is_school() && $organization->get_remaining_licenses() <= 0 ) {
				return new WP_REST_Response(
					array(
						'success' => false,
						'message' => 'Organization is out of licenses',
					),
					403
				);
			}

			$user = email_exists( $user_data['email'] );

			if ( ! $user ) {
				$user_data['user_pass']  = wp_generate_password();
				$user_data['user_login'] = $user_data['user_email'];
				$new_user_id             = wp_insert_user( $user_data );
				if ( is_wp_error( $new_user_id ) ) {
					return new WP_REST_Response(
						array(
							'success' => false,
							'message' => $new_user_id->get_error_message(),
						),
						500
					);
				}

				wp_new_user_notification( $new_user_id );
				$user = get_user_by( 'ID', $new_user_id );

			} else {
				$has_organizations = get_users_organization_id( $user );

				if ( ! empty( $has_organizations ) && $has_organizations[0] !== $organization->get_organization()->ID ) {
					return new WP_REST_Response(
						array(
							'success' => false,
							'message' => 'User already belongs to an organization.',
						),
						400
					);
				}

				$user = get_user_by( 'email', $user_data['email'] );
			}

			$organization->add_user_to_organization( $user->ID, 'group_leaders' );

			if ( $organization->is_school() ) {
				$user->add_role( 'opettaja' );
			}
			$user->add_role( 'group_leader' );

			ld_update_leader_group_access( $user->ID, $group_id );

			$group = get_post( $group_id );

			$email_data = array(
				'group_name' => $group->post_title,
				'user_name'  => $user->first_name,
			);

			send_email( 'Kädenjälki <noreply@kadenjalki.fi>', $user->user_email, $email_data, 'new-group-leader' );
			$response = new WP_REST_Response(
				array(
					'success' => true,
					'message' => 'Added new group leader.',
				),
				200,
			);

			TransientManager::delete( $group_id . 'group_group_leaders' );

		} catch ( PermissionException | OrganizationNotFoundException | InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				404
			);
		}

		return rest_ensure_response( $response );
		// phpcs:enable Squiz.PHP.NonExecutableCode.Unreachable
	}

	/**
	 * Handles either creating or adding an existing user to a group and letting them know.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response The response object.
	 */
	final public function add_user_to_group( WP_REST_Request $request ): WP_REST_Response {
		$group_id = $request->get_param( 'group_id' );
		$org_id   = $request->get_param( 'id' );

		$user_data = array(
			'email'      => $request->get_param( 'user_email' ),
			'user_email' => $request->get_param( 'user_email' ),
			'first_name' => $request->get_param( 'first_name' ),
			'last_name'  => $request->get_param( 'last_name' ),
		);

		try {
			$organization = Organization::with_id( $org_id );

			if ( ! $organization->has_organization() ) {
				throw new OrganizationNotFoundException();
			}

			if ( ! $organization->is_school() && $organization->get_remaining_licenses() <= 0 ) {
				return new WP_REST_Response(
					array(
						'success' => false,
						'message' => 'Organization is out of licenses',
					),
					403
				);
			}

			$user     = email_exists( $user_data['email'] );
			$existing = false;
			if ( ! $user ) {
				$user_data['user_pass']  = wp_generate_password();
				$user_data['user_login'] = $user_data['user_email'];
				$new_user_id             = wp_insert_user( $user_data );
				if ( is_wp_error( $new_user_id ) ) {
					return new WP_REST_Response(
						array(
							'success' => false,
							'message' => $new_user_id->get_error_message(),
						),
						500
					);
				}

				wp_new_user_notification( $new_user_id, null, 'user' );
				$user = get_user_by( 'ID', $new_user_id );

			} else {
				$user     = get_user_by( 'email', $user_data['email'] );
				$existing = true;
			}

			ld_update_group_access( $user->ID, $group_id );

			$email_data = array(
				'first_name'     => $user->first_name,
				'site_url'       => site_url(),
				'enrollment_url' => get_signed_enrollment_url( $group_id ),
			);

			send_email( 'Kädenjälki <noreply@kadenjalki.fi>', $user->user_email, $email_data, $existing ? 'group-invite-existing' : 'group-invite-new' );

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'message' => 'Invited user to group.',
				),
				200,
			);

			$transient_name = $group_id . '_group_users_progress';

			TransientManager::delete( $transient_name );

			if ( $request->has_param( 'course_id' ) ) {
				$transient_name .= '_course_' . $request->get_param( 'course_id' );
				TransientManager::delete( $transient_name );
			}

		} catch ( OrganizationNotFoundException | InvalidArgumentException $e ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $e->getMessage(),
				),
				404
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * GETTERS
	 */

	/**
	 * Gets the namespace string for the Rest API.
	 *
	 * @return string
	 */
	final public function get_namespace(): string {
		return $this->namespace . '/' . $this->version;
	}

	/**
	 * Returns the array of valid roles that can be requested.
	 *
	 * @return string[]
	 */
	private function get_valid_roles_array(): array {
		return array( 'group_leader', 'organization_owner' );
	}

	/**
	 * Registers the routes for handling organization stuff.
	 *
	 * @return void
	 */
	final public function register_organization_routes(): void {
		register_rest_route(
			$this->get_namespace(),
			'search/(?P<searchString>.+)',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'search_user' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_logged_in' ),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_organization' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_logged_in' ),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_groups' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_owner' ),
				'args'                => array( 'id' => RestAPIValidationHelpers::validate_id_argument() ),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/courses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_organization_courses' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_owner' ),
				'args'                => array( 'id' => RestAPIValidationHelpers::validate_id_argument() ),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/new_courses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_organization_allowed_courses' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader' ),
				'args'                => array(
					'id'       => RestAPIValidationHelpers::is_valid_org_id(),
					'group_id' => RestAPIValidationHelpers::is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/courses',
			array(
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => array( $this, 'add_course_to_organization_group' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader' ),
				'args'                => array(
					'id'        => RestAPIValidationHelpers::is_valid_org_id(),
					'group_id'  => RestAPIValidationHelpers::is_valid_ld_group_id(),
					'course_id' => RestAPIValidationHelpers::is_valid_ld_course_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/enrollment',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_group_enrollment_url' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader' ),
				'args'                => array(
					'id'       => RestAPIValidationHelpers::is_valid_org_id(),
					'group_id' => RestAPIValidationHelpers::is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/licenses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_licenses' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_owner' ),
				'args'                => array( 'id' => RestAPIValidationHelpers::validate_id_argument() ),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/invite',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'invite_user_to_group' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader' ),
				'args'                => array(
					'group_id' => RestAPIValidationHelpers::is_valid_ld_group_id(),
					'id'       => RestAPIValidationHelpers::is_valid_org_id(),
					'email'    => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param ) && false !== filter_var( trim( $param ), FILTER_VALIDATE_EMAIL );
						},
						'sanitize_callback' => function ( $param, $request, $key ) {
							return sanitize_email( $param );
						},
					),
				),
				'rate_limiter'        => array(
					'uses'     => 5,
					'interval' => 60,
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/available_leaders',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_available_group_leaders' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader' ),
				'args'                => array(
					'id'       => RestAPIValidationHelpers::is_valid_org_id(),
					'group_id' => RestAPIValidationHelpers::is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/add_leader',
			array(
				'methods'             => WP_REST_Server::EDITABLE,
				'callback'            => array( $this, 'add_leader_to_group' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader' ),
				'args'                => array(
					'group_id' => RestAPIValidationHelpers::is_valid_ld_group_id(),
					'id'       => RestAPIValidationHelpers::is_valid_org_id(),
					'user_id'  => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_numeric( $param ) && false !== get_user_by( 'ID', $param );
						},
					),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/add_leader_deprecated',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'invite_group_leader' ),
				'permission_callback' => '__return_false',
				'args'                => array(
					'group_id'   => RestAPIValidationHelpers::is_valid_ld_group_id( array() ),
					'id'         => RestAPIValidationHelpers::is_valid_org_id(),
					'user_email' => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param ) && false !== filter_var( trim( $param ), FILTER_VALIDATE_EMAIL );
						},
						'sanitize_callback' => function ( $param, $request, $key ) {
							return sanitize_email( $param );
						},
					),
					'first_name' => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param );
						},
						'sanitize_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
					'last_name'  => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param );
						},
						'sanitize_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'org/(?P<id>\d+)/groups/(?P<group_id>\d+)/add_user',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'add_user_to_group' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader' ),
				'args'                => array(
					'group_id'   => RestAPIValidationHelpers::is_valid_ld_group_id( array() ),
					'course_id'  => RestAPIValidationHelpers::is_valid_ld_course_id(),
					'id'         => RestAPIValidationHelpers::is_valid_org_id(),
					'user_email' => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param ) && false !== filter_var( trim( $param ), FILTER_VALIDATE_EMAIL );
						},
						'sanitize_callback' => function ( $param, $request, $key ) {
							return sanitize_email( $param );
						},
					),
					'first_name' => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param );
						},
						'sanitize_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
					'last_name'  => array(
						'required'          => true,
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param );
						},
						'sanitize_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
				),
			)
		);
	}

	/**
	 * Registers the routes for groups.
	 *
	 * @return void
	 */
	final public function register_groups_routes(): void {
		register_rest_route(
			$this->get_namespace(),
			'groups',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_requesting_users_groups' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_logged_in' ),
				'args'                => array(
					'role' => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return is_string( $param ) && in_array( (string) $param, $this->get_valid_roles_array(), true );
						},
					),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/course/(?P<course_id>\d+)/information',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_extended_course_information' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader_of_post' ),
				'args'                => array(
					'course_id' => RestAPIValidationHelpers::is_valid_ld_course_id(),
					'group_id'  => RestAPIValidationHelpers::is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/extra',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_extra_group_information' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id' => RestAPIValidationHelpers::is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/courses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_groups_course_ids' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id' => RestAPIValidationHelpers::is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/users',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_groups_user' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id'  => RestAPIValidationHelpers::is_valid_ld_group_id(),
					'course_id' => RestAPIValidationHelpers::is_valid_ld_course_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/group_leaders',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_group_leaders' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id' => RestAPIValidationHelpers::is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/course/(?P<course_id>\d+)/quizzes',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_course_quizzes' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id'  => RestAPIValidationHelpers::is_valid_ld_group_id(),
					'course_id' => RestAPIValidationHelpers::is_valid_ld_course_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/course/(?P<course_id>\d+)/quiz/(?P<quiz_id>\d+)',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_group_course_quiz' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id'  => RestAPIValidationHelpers::is_valid_ld_group_id(),
					'course_id' => RestAPIValidationHelpers::is_valid_ld_course_id(),
					'quiz_id'   => RestAPIValidationHelpers::is_valid_ld_quiz_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'groups/(?P<group_id>\d+)/course/(?P<course_id>\d+)/quiz/(?P<quiz_id>\d+)/attempts/',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_quiz_data' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'is_group_leader_of_post' ),
				'args'                => array(
					'group_id'  => RestAPIValidationHelpers::is_valid_ld_group_id( array() ),
					'course_id' => RestAPIValidationHelpers::is_valid_ld_course_id(),
					'quiz_id'   => RestAPIValidationHelpers::is_valid_ld_quiz_id(),
					'user_id'   => RestAPIValidationHelpers::is_valid_user_id(),
				),
			)
		);
	}

	/**
	 * Registers the user routes.
	 *
	 * @return void
	 */
	final public function register_user_routes(): void {
		register_rest_route(
			$this->get_namespace(),
			'user/(?P<user_id>\d+)/groups',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_users_groups' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'can_manage_requested_user' ),
				'args'                => array(
					'user_id' => RestAPIValidationHelpers::is_valid_user_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'user/(?P<user_id>\d+)/groups/(?P<group_id>\d+)/course_progress',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_user_group_progress' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'can_manage_requested_user' ),
				'args'                => array(
					'user_id'  => RestAPIValidationHelpers::is_valid_user_id(),
					'group_id' => RestAPIValidationHelpers::is_valid_ld_group_id(),
				),
			)
		);
		register_rest_route(
			$this->get_namespace(),
			'user/(?P<user_id>\d+)/certificates',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_user_certificates' ),
				'permission_callback' => array( RestAPIPermissionHelpers::class, 'can_manage_requested_user' ),
				'args'                => array(
					'user_id' => RestAPIValidationHelpers::is_valid_user_id(),
				),
			)
		);
	}
}
