<?php
/**
 * Handles registering the RestAPI endpoints for handling organizations.
 *
 * @package LD_Organization
 */

namespace LD_Organization;

use Stripe\Exception\InvalidArgumentException;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * RestAPI class, when initialized will register the restAPI.
 *
 * @package LD_Organization
 * @since 0.2.0
 */
class RestAPI {

	/**
	 * The RestAPI namespace.
	 *
	 * @var string
	 */
	private string $namespace = 'ld-organization';

	/**
	 * The RestAPI version.
	 *
	 * @var string
	 */
	private string $version = 'v1';

	/**
	 * Registers all required functionality and endpoints.
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_endpoints' ) );
	}

	/**
	 * The default validate_callback if the endpoint has an ID in it.
	 *
	 * @return array
	 */
	final public function validate_id_argument(): array {
		return array(
			'id' => array(
				'validate_callback' => function( $param, $request, $key ) {
					return is_numeric( $param );
				},
			),
		);
	}

	/**
	 * Registers all the endpoint and their parameters.
	 *
	 * @return void
	 */
	final public function register_endpoints(): void {
		register_rest_route(
			$this->namespace . '/' . $this->version,
			'org/(?P<id>\d+)/licenses',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_licenses' ),
				'permission_callback' => array( $this, 'is_owner' ),
				'args'                => $this->validate_id_argument(),
			)
		);
	}

	/**
	 * Checks if the user calling the API is an owner of the organization being called.
	 *
	 * @param WP_REST_Request $request The request object for the current request.
	 *
	 * @uses \LD_Organization\RestAPI::has_access()
	 * @return bool
	 */
	final public function is_owner( WP_REST_Request $request ): bool {
		return $this->has_access( $request, 'organization_owner' );
	}

	/**
	 * Checks if the user calling the API is an owner of the organization being called.
	 *
	 * @param WP_REST_Request $request The request object for the current request.
	 *
	 * @uses \LD_Organization\RestAPI::has_access()
	 * @return bool
	 */
	final public function is_group_leader( WP_REST_Request $request ): bool {
		return $this->has_access( $request, 'group_leaders' );
	}

	/**
	 * Handles checking if the organization is owned by the authenticated user.
	 *
	 * @param WP_REST_Request $request The request with a required parameter of ID.
	 * @param string          $role The role to check against.
	 *
	 * @uses \LD_Organization\Organization::with_id(),\LD_Organization\Organization::has_user()
	 * @return bool
	 */
	final public function has_access( WP_REST_Request $request, string $role = 'both' ): bool {
		if ( ! is_user_logged_in() ) {
			return false;
		}
		$id = $request->get_param( 'id' );
		if ( ! isset( $id ) ) {
			return false;
		}
		try {
			$organization = Organization::with_id( $id );
			$current_user = wp_get_current_user();
			if ( isset( $current_user ) && $organization->has_user( $current_user->ID, $role ) ) {
				return true;
			}
		} catch ( OrganizationNotFoundException | InvalidArgumentException $exception ) {
			return false;
		}
		return false;
	}

	/**
	 * Returns the requested organizations licenses.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @uses \LD_Organization\Organization::with_id(),\LD_Organization\Organization::get_licenses(),\LD_Organization\Organization::get_used_licenses()
	 * @return WP_REST_Response
	 */
	final public function get_licenses( WP_REST_Request $request ): WP_REST_Response {
		try {
			$id           = $request->get_param( 'id' );
			$organization = Organization::with_id( $id );

			$used_licenses   = $organization->get_used_licenses();
			$unused_licenses = $organization->get_licenses();

			$response = new WP_REST_Response(
				array(
					'success' => true,
					'data'    => array(
						'used_licenses'   => $used_licenses,
						'unused_licenses' => $unused_licenses,
					),
					'message' => 'Successfully fetched licenses for organization' . $id,
				)
			);

		} catch ( InvalidArgumentException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => $exception->getMessage(),
				),
				400
			);
		} catch ( OrganizationNotFoundException $exception ) {
			$response = new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Organization not found',
				),
				400
			);
		}
		return rest_ensure_response( $response );
	}
}
