<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName, WordPress.Files.FileName.NotHyphenatedLowercase
// phpcs:disable Generic.Commenting.DocComment.MissingShort

namespace LD_Organization\Hooks;

use InvalidArgumentException;
use LD_Organization\Exceptions\OrganizationNotFoundException;
use LD_Organization\Organization;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Handles the validation for saving an organization so that all the different fields are valid.
 *
 * @package LD_Organization\Hooks
 * @since 0.14.0 Added.
 */
class OrganizationCustomFields extends AbstractHook {

	/**
	 * Handles the validation for saving an organization owner.
	 *
	 * @param bool|string $valid If the value is valid (bool) or a custom error message.
	 * @param mixed       $value The value of the field.
	 * @param array       $field The field settings.
	 * @param string      $input_name The name of the input field (DOM).
	 *
	 * @return bool|string
	 */
	final public function validate_user_field( $valid, $value, array $field, string $input_name ) {

		// Return early if we already have an invalid value.
		if ( true !== $valid ) {
			return $valid;
		}

		if ( is_array( $value ) && ! empty( $_POST['post_id'] ) ) { // @phpcs:ignore WordPress.Security.NonceVerification.Missing
			$users_error_string = '';
			$error              = '';

			$post_id = absint( $_POST['post_id'] ); // @phpcs:ignore WordPress.Security.NonceVerification.Missing

			// Unoptimized, but not sure if you can query more than one user ID at a time.
			foreach ( $value as $user_id ) {
				$args   = array(
					'post_type'   => Organization::$post_type,
					'numberposts' => 1,
					'exclude'     => array( $post_id ),
					'meta_query'  => array( // @phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
						'relation' => 'OR',
						array(
							'key'     => 'organization_owners',
							'value'   => '"' . $user_id . '"',
							'compare' => 'LIKE',
						),
						array(
							'key'     => 'group_leaders',
							'value'   => '"' . $user_id . '"',
							'compare' => 'LIKE',
						),
					),
					'fields'      => 'ids',
				);
				$result = get_posts( $args );
				if ( ! empty( $result ) ) {
					$user = get_user_by( 'id', $user_id );
					if ( ! empty( $user ) ) {
						$title = get_the_title( $result[0] );
						if ( empty( $users_error_string ) ) {
							$users_error_string = $user->display_name . ' (' . $title . ')';
							continue;
						}
						$users_error_string .= ', ' . $user->display_name . ' (' . $title . ')';
					}
				}
			}
			if ( ! empty( $users_error_string ) ) {
				/* translators: %s a list of users that are found in another organization */
				return sprintf( __( 'Users cannot be added to multiple organizations: %s', 'ld-organization' ), $users_error_string );
			}
		}

		return true;
	}

	/**
	 * Handles filtering the Owner user field so that only the correct role users are shown.
	 *
	 * @param array $args The arguments passed to get_users() function.
	 * @param mixed $field THe field options passed to the filter.
	 * @param int   $post_id The post ID of the post that the field is currently in.
	 *
	 * @return array
	 * @see get_users For an example of the $args.
	 */
	final public function filter_user_field_owner( array $args, $field, int $post_id ): array {
		$args['role__in'] = Organization::WP_ROLE_NAME_OWNERS;

		return $args;
	}

	/**
	 * Handles filtering the Owner user field so that only the correct role users are shown.
	 *
	 * @param array $args The arguments passed to get_users() function.
	 * @param mixed $field THe field options passed to the filter.
	 * @param int $post_id The post ID of the post that the field is currently in.
	 *
	 * @return array
	 * @see get_users For an example of the $args.
	 */
	final public function filter_user_field_group_leaders( array $args, $field, int $post_id ): array {
		$args['role__in'] = Organization::WP_ROLE_NAME_GROUP_LEADERS;

		return $args;
	}

	/**
	 * Handles the validation for license field.
	 *
	 * @param bool|string $valid If the value is valid (bool) or a custom error message.
	 * @param mixed $value The value of the field.
	 * @param array $field The field settings.
	 * @param string $input_name The name of the input field (DOM).
	 *
	 * @return bool|string
	 */
	final public function validate_license_field( $valid, $value, array $field, string $input_name ) {
		// Return early if we already have an invalid value.
		if ( true !== $valid ) {
			return $valid;
		}

		if ( ! empty( $_POST['post_id'] ) ) { // @phpcs:ignore WordPress.Security.NonceVerification.Missing
			$post_id = absint( $_POST['post_id'] ); // @phpcs:ignore WordPress.Security.NonceVerification.Missing

			try {
				$organization = Organization::with_id( $post_id );
				$value        = absint( $value );
				if ( ! $organization->has_organization() ) {
					throw new OrganizationNotFoundException();
				}
				if ( $organization->is_school() ) {
					return $valid;
				}

				$used_licenses = $organization->get_used_licenses( true );

				if ( $used_licenses > $value ) {
					return __( 'Licenses cannot be less than used licenses, please remove users from the linked groups.', 'ld-organization' );
				}
			} catch ( OrganizationNotFoundException $e ) {
				return $valid;
			} catch ( InvalidArgumentException $e ) {
				return $e->getMessage();
			}
		}

		return $valid;
	}

	/**
	 * @inheritDoc
	 */
	public function get_actions(): array {
		return array();
	}

	/**
	 * @inheritDoc
	 */
	public function get_filters(): array {
		return array(
			array(
				'hook'     => 'acf/fields/user/query/key=field_629073ab801c2',
				'callable' => array( $this, 'filter_user_field_owner' ),
				'priority' => 10,
				'num_args' => 4,
			),
			array(
				'hook'     => 'acf/validate_value/key=field_629073ab801c2',
				'callable' => array( $this, 'validate_user_field' ),
				'priority' => 10,
				'num_args' => 4,
			),
			array(
				'hook'     => 'acf/fields/user/query/key=field_6290740d801c3',
				'callable' => array( $this, 'filter_user_field_group_leaders' ),
				'priority' => 10,
				'num_args' => 4,
			),
			array(
				'hook'     => 'acf/validate_value/key=field_6290740d801c3',
				'callable' => array( $this, 'validate_user_field' ),
				'priority' => 10,
				'num_args' => 4,
			),
			array(
				'hook'     => 'acf/validate_value/key=field_62907364801c1',
				'callable' => array( $this, 'validate_license_field' ),
				'priority' => 10,
				'num_args' => 4,
			),
		);
	}
}
