<?php

namespace LD_Organization\Cron;

use Monolog\Logger;
use const LD_Organization\LD_ORG_BASE_PATH;
use const LD_Organization\LD_ORG_PROCESSING_TRANSIENT_KEY;

/**
 * Class CleanupExportFiles
 *
 * @package LD_Organization\Cron
 * @since 0.20.0
 */
class CleanupExportFiles extends AbstractCronJob {

	/**
	 * The logger instance.
	 *
	 * @var Logger
	 */
	private Logger $logger;

	/**
	 * @inerhitDoc
	 */
	public function __construct( $interval, $args ) {
		parent::__construct( $interval, $args );
		$this->logger = logger( 'CleanupExportFiles' );
	}

	/**
	 * Handles checking if there are any export files that are older than the specified time.
	 *
	 * @return void
	 */
	final public function do_cron_job(): void {
		$this->logger->info( 'Cleaning up export files.', $this->cron_args );
		$files = glob( trailingslashit( LD_ORG_BASE_PATH ) . 'export/*.csv' );
		$this->logger->debug( 'Found files.', array( 'files' => $files ) );
		if ( empty( $files ) ) {
			$this->logger->info( 'No files found.', $this->cron_args );

			return;
		}
		$transients = get_transient( LD_ORG_PROCESSING_TRANSIENT_KEY );
		$this->logger->debug( 'Found transients.', array( 'transients' => $transients ) );
		if ( empty( $transients ) ) {
			foreach ( $files as $file ) {
				wp_delete_file( $file );
			}
			$this->logger->info( 'Deleted files.', $this->cron_args );

			return;
		}
		foreach ( $files as $file ) {
			$file_name = pathinfo( $file, PATHINFO_FILENAME );
			if ( empty( $file_name ) ) {
				continue;
			}
			$id = str_replace( array( 'export-', '.csv' ), '', $file_name );
			if ( empty( $id ) ) {
				$this->logger->info(
					'File name does not contain an ID. Deleting file.',
					array(
						'file' => $file,
					)
				);
				wp_delete_file( $file );
				continue;
			}
			if ( ! in_array( $id, $transients, true ) ) {
				$this->logger->info(
					'File is not in transient. Deleting file.',
					array(
						'file'      => $file,
						'transient' => $transients,
					)
				);
				wp_delete_file( $file );
			}
		}
	}
}
