#!/usr/bin/env node
import * as esbuild from "esbuild";
import { sassPlugin } from "esbuild-sass-plugin";
import postcss from "postcss";
import autoprefixer from "autoprefixer";
import postcssPresetEnv from "postcss-preset-env";
import { readdirSync, readFileSync, existsSync } from "fs";
import { join, parse } from "path";
const buildData = {
	scripts: [
		{
			entryPoints: ["assets/js/"],
			outdir: "dist/js",
		},
	],
	styles: [
		{
			entryPoints: ["assets/scss"],
			outdir: "dist/css",
		},
	],
};

const options = {
	bundle: true,
	minify: true,
	write: true,
	sourcemap: false,
	logLevel: "info",
	entryNames: "[name]",
	external: [
		"*.png",
		"*.svg",
		"*.jpg",
		"*.jpeg",
		"*.css",
		"*.woff",
		"*.woff2",
		"*.otf",
		".gitkeep",
	],
};

const sassOptions = {
	async transform(source, resolveDir) {
		const { css } = await postcss([
			autoprefixer,
			postcssPresetEnv({ stage: 0 }),
		]).process(source, { from: "dist/css" });
		return css;
	},
};

// Scripts
for (const script of buildData.scripts) {
	const entries = [];
	for (const entry of script.entryPoints) {
		entries.push(...getFiles(entry));
	}
	const config = {
		...options,
		entryPoints: deDupe(entries),
		loader: { ".js": "jsx" },
		outdir: script.outdir,
	};
	await build(config);
}

// Styles
for (const style of buildData.styles) {
	const entries = [];
	for (const entry of style.entryPoints) {
		entries.push(...getFiles(entry));
	}
	const config = {
		...options,
		entryPoints: deDupe(entries),
		plugins: [sassPlugin(sassOptions)],
		outdir: style.outdir,
	};
	await build(config);
}

async function build(config) {
	if (process.argv[2] === "build") {
		await esbuild.build(config);
	} else if (process.argv[2] === "watch") {
		// ESbuild 0.16 syntax used until dependencies are resolved for 0.17.
		await esbuild.build({ ...config, watch: true });
		// ESbuild 0.17 syntax.
		//await (await esbuild.context(config)).watch();
	}
}

/**
 * Get all files that needs to be passed to esbuild from the given path.
 *
 * @param path
 * @returns {*[]}
 */
function getFiles(path) {
	const files = [];
	if (existsSync(path)) {
		for (const file of readdirSync(path)) {
			// Only add files not starting with _ and that has an extension.
			if (file.substring(0, 1) !== "_" && file.split(".").length > 1) {
				files.push(join(path, file));
			}
		}
	}
	return files;
}

/**
 * Remove duplicate filenames from list to allow "overwrites".
 * @param files array<string>
 * @return array<string>
 */
function deDupe(files) {
	const names = [];
	return files.filter((file) => {
		const path = parse(file);
		if (!names.includes(path.name)) {
			names.push(path.name);
			return true;
		}
		return false;
	});
}
