<?php


namespace RateLimiter;

class Client {

	/**
	 * The client id.
	 *
	 * @var string|int
	 */
	private $id;

	/**
	 * The interval (in seconds) before the usage timer is reset.
	 *
	 * @var int
	 */
	private $interval;

	/**
	 * Maximum allowed uses during an interval.
	 *
	 * @var int
	 */
	private $max_uses;

	/**
	 * The amount of uses during a interval.
	 *
	 * @var int
	 */
	private $uses;

	/**
	 * Start time of an interval.
	 *
	 * @var
	 */
	private $start;


	/**
	 * Client constructor.
	 */
	public function __construct( $id, $max_uses, $interval ) {
		$this->id       = $id;
		$this->max_uses = $max_uses;
		$this->interval = $interval;
		$this->uses     = $max_uses;
		$this->start    = time();
	}

	/**
	 * Adds a number of uses to this client.
	 *
	 * @param int $times The amount of uses to add.
	 * @return Client
	 */
	public function add_use( $times = 1 ) {
		$this->uses -= $times;
		return $this;
	}

	/**
	 * Returns the clients id.
	 *
	 * @return int|string
	 */
	public function get_id() {
		return $this->id;
	}

	/**
	 * Returns the interval for this client.
	 *
	 * @return int
	 */
	public function get_interval() {
		return $this->interval;
	}

	/**
	 * Returns the max uses for the client.
	 *
	 * @return int
	 */
	public function get_max_uses() {
		return $this->max_uses;
	}

	/**
	 * Returns the clients current uses during this interval.
	 *
	 * @return int
	 */
	public function get_uses() {
		return $this->uses;
	}

	/**
	 * Returns the start time for this interval.
	 *
	 * @return mixed
	 */
	public function get_start() {
		return $this->start;
	}

	/**
	 * Returns when this interval will reset.
	 *
	 * @return int
	 */
	public function get_reset() {
		return $this->start + $this->interval - time();
	}

	/**
	 * Whether the max uses has been exceeded.
	 *
	 * @return bool
	 */
	public function is_max_uses_exceeded() {
		return 0 > $this->get_uses();
	}

	/**
	 * Retrieve rate limit headers.
	 *
	 * @return array
	 */
	public function get_headers() {
		$headers = array(
			'X-RateLimit-Limit'     => $this->get_max_uses(),
			'X-RateLimit-Remaining' => $this->get_uses(),
			'X-RateLimit-Reset'     => $this->get_reset(),
		);

		if ( $this->is_max_uses_exceeded() ) {
			$headers['Retry-After']           = $this->get_reset();
			$headers['X-RateLimit-Remaining'] = 0;
		}

		return $headers;
	}

	/**
	 * Converts the client to an array.
	 *
	 * @return array
	 */
	public function to_array() {
		return array(
			'max_uses'  => $this->get_max_uses(),
			'remaining' => $this->is_max_uses_exceeded() ? 0 : $this->get_max_uses(),
			'reset'     => $this->get_reset(),
		);
	}
}
